## ----chunkOptions,echo=FALSE--------------------------------------------------
#set global chunk options
opts_chunk[["set"]](dev='tikz', external=TRUE) #has to be set to true as I work with standalone for thesis
opts_chunk[["set"]](warning=FALSE,error=TRUE,message=FALSE,echo=FALSE,tidy=FALSE)
opts_chunk[["set"]](dev='tikz',external=TRUE,fig.width=5,fig.height=3)
opts_chunk[["set"]](fig.align='center',fig.show='hold')
opts_chunk[["set"]](cache=TRUE,autodep=TRUE, par=TRUE, comment=NA, keep.blank.line=FALSE)
knit_hooks[["set"]](par=function(before, options, envir){
 if (before) { options(max.print=40) ; options(width=80) }
 if (before && options[["fig.show"]]!='none') par(mar=c(4,4,0,.1),cex.main=1,font.main=1,mgp=c(2,.7,0),tcl=-.3)
}, crop=hook_pdfcrop)
knit_hooks$set(document = function(x) {sub('\\usepackage[]{color}', '\\usepackage{xcolor}', x, fixed = TRUE)}) #\WM{To circumvent the constent warning in latex}

## ----setup, include=FALSE, cache=FALSE----------------------------------------
#doBayes=FALSE
#libraries used in this document
library(knitr)
library(plyr) 
library(psy)      ## cronbach
library(lavaan)   ## used for sem
library(psych)    ## corr.test
library(lme4)     ## lmer
library(gamlss)   ## splines
library(mgcv)     ## gamm
library(lmtest)   ## lrtest
library(boot)
library(bootstrap)## jackknife
library(reshape)  ## melt in preparation of data
library(zTree)    ## zTreeTables
library(ggplot2)
library(stargazer)
#
library(lattice)
library(latticeExtra)
library(RColorBrewer) ## no longer automatically loaded with latticeExtra
library(xtable)
##
library(parallel)
options(mc.cores=detectCores())
#
ltheme <- canonical.theme(color = FALSE)
ltheme$strip.background$col <- "transparent"
ltheme$par.main.text$font=1
ltheme$par.main.text$cex=1
ltheme$par.sub.text$font=1
lattice.options(default.theme = ltheme)
#
mPal<-brewer.pal(6,"Greys")
mTheme<-custom.theme(fill = mPal)
mTheme$par.main.text$font=1
mTheme$par.main.text$cex=1
mTheme$par.sub.text$font=1
#
options(xtable.floating=FALSE)
options(xtable.sanitize.text.function=function(x) x)
#


## ----ReadingData,error=FALSE,include=FALSE,warning=FALSE----------------------
# --------------------------------------------------------------------------------#
# ---------------------- Reading the ztree Date into R ---------------------------#
# --------------------------------------------------------------------------------#
options(zTree.silent=TRUE) # <- less chatty
setwd("../Data/Daten")
allFiles<-list.files(".","*.xls",recursive=TRUE)
zTree       <- zTreeTables( allFiles )
subjects <- zTree$subjects
globals <- zTree$globals 
#um Subject files (also questionnaire zu buchen)
files <- list.files(pattern = "*.sbj$",recursive=TRUE)
fname <- sub(".*/","",files)
sbj <- zTreeSbj(files) 
setwd("../../Paper")
sbj<-sbj[order(sbj$Date,as.numeric(as.character(sbj$Subject))),]
# ---------------------- Alternative if no internet ------------------------------#
#save(subjects,globals,sbj, file = "subjects.RData")
#load( "subjects.RData")
# --------------------------------------------------------------------------------#


## ----DataPreparation,error=FALSE,include=FALSE,warning=FALSE------------------
#We always ran the SVO+Risk "Treatment" first (subjects$Treat_Name==1)
#We ran the Auction and Spite-Auction "Treatment" second (subjects$Treat_Name==2)
#The payoff presentation "treatment" was always last (subjects$Treat_Name==3)
#
#In the first session we had 20 rounds and we had to abort the testquestions for the 
#spite-auction measure 
#
#we counterbalanced the Auction with the Spite-Auction Measure therefore we have to make a 
#distinction whether the all-pay auction  was first or second 
#
# first we need the matching groups
# unfortunately this has not been coded already in z-Tree
# the following does most of the matching groups:
subjects<-within(subjects,{sid<-as.factor(paste(Date,Subject,sep="_"))
                     gid<-(Subject+5) %/% 6})
# we NEED this information also for sbj
sbj<-within(sbj,{sid<-as.factor(paste(Date,Subject,sep="_"))})
# there was one experiment where one of these groups would be too small
gpcorrect<-subset(with(subjects,aggregate(Subject,list(Date=Date,gid=gid,gp=paste(Date,gid)),function(x) length(unique(x)))),x<4)
xx2<-join(subjects,within(gpcorrect,gpc<- -1))
subjects[!is.na(xx2$gpc),"gid"]<-subjects[!is.na(xx2$gpc),"gid"]-1
subjects<-within(subjects,gid<-paste(Date,gid,sep="_"))
#
subjects$Date<-as.numeric(gsub("_","",subjects$Date)) #change data-string
globals$Date<-as.numeric(gsub("_","",globals$Date)) #change data-string
sbj$Date<-as.numeric(gsub("_","",sbj$Date))
# ---------------------- Assinging the firstprice var.  --------------------------#
subjects$FirstPrice<-ifelse(subjects$Date>1705010745,1,0) #experiments will start may 2nd 2017
sbj$FirstPrice<-ifelse(sbj$Date>1705010745,1,0)
#
Runden<-15#the amount of rounds
Nsubj<-nrow(sbj[sbj$FirstPrice==0,]) #the amount of participated subjects
Nsubj2<-nrow(sbj[sbj$FirstPrice==1,]) #the amount of participated subjects
Nsess<-length(unique(subset(subjects,FirstPrice==0)[["Date"]])) ## number of sessions
costs<-sum(subjects$Profit_overall2[subjects$Treat_Name==3 & subjects$FirstPrice==0])
# ---------------------- Calculating scores etc for sbj --------------------------#
#Calculating optimism
toNum<-grep("^(Spite|Rivalry|Optimism)[0-9]+$",names(sbj))
for(i in toNum) sbj[,i]<-as.numeric(sbj[,i])
rivNames<-grep("^Rivalry[0-9]+$",names(sbj),value=TRUE)
#Calculating rivalry
sbj$rivalry<-rowMeans(sbj[,rivNames[c(4,6,9,10:14,17)]])
#Calculating admiration
sbj$admiration<-rowMeans(sbj[,rivNames[c(1,2,3,5,7,8,15,16,18)]])
#Calculating rivalry_devaluation
sbj$rivalry_deval<-rowMeans(sbj[,rivNames[c(13,14,17)]])
#Calculating rivalry_supremacy
sbj$rivalry_suprem<-rowMeans(sbj[,rivNames[c(6,9,10)]])
#Calculating rivalry_agrressiveness
sbj$rivalry_agress<-rowMeans(sbj[,rivNames[c(4,11:12)]])
#Calculating spitefullness
spiteQuestVal<-rbind(1,1,-1,1,1,1,1,1,1,1,1,1,1,-1,1,1,1)
sbj$Spite_Quest<-((12 + as.matrix(sbj[,grep("^Spite[0-9]+$",names(sbj))]) %*% spiteQuestVal)/17)
#recoding female and male into english
sbj<-within(sbj,{Geschlecht<-ifelse(Geschlecht=="Weiblich","Female","Male")})
# --------------------------------------------------------------------------------#
# ---------------------- Auction inherents only the Auction Data -----------------#
# ------------- As we counterbalanced the Auction with the Spite Measure -----------------#
# ------------- we have to make a distinction whether the all-pay auction  ---------------#
# -------------                was first or second                         ---------------#
#here we take the case when the all-pay auction was played in period 2-16 (note that in the first session
#we had 20 rounds and therefore, we have to restrict period to 16)
# *** add Runden to subjects:     
subjects<-join(subjects,globals[,c("Date","Period","Treatment","Runden")])
xx<-within(with(subset(subjects,Period==Runden),aggregate(Treatment~Date,FUN=median)),{
       AuctionTreat<-TRUE;})
subjects<-join(subjects,xx) 
subjects<-within(subjects,{
    KimAuction<-AuctionTreat & ((AllPay_First==1 & Period==Runden+1 ) | (AllPay_First==0 & Period==1 ))
    APAuction <- AuctionTreat & !KimAuction})

Auction<-subset(subjects,APAuction)[,c("Date","Subject","sid","gid","Period",grep("^Choice\\[[0-9]+\\]$",names(subjects),value=TRUE),"Dauer_Stage1AllPay","Dauer_Stage1AllPay_fb","FirstPrice")]
##
names(Auction)<-gsub("\\[|\\]","",names(Auction))
# ---------------------- Personality-Data is containing:         -----------------#
Personality<-subset(subjects,Treat_Name==1)[,grep("^sid$|^svo|^inequality|^Spitefullness|^Risk[1-9_]",names(subjects),value=TRUE)]
Personality<-within(Personality,Risk_overall <- 10 - Risk_overall) # higher values represent higher risk aversion
Personality<-rename(Personality,c("svo_angle"="SVO"))
#In the spite measure so far I just have looked how stron one deviates from the pareto-option - however, I have not taken into account how many points were deducted.
#Now I'm defining spite as the amount of points deducted from the other player
SpiteTableSVO<-t(apply(read.table(text='
70 100 70 100 100 100 100 100 100 70  85 50
70  85 50  70 100  85  85  70 100 85 100 85'),2,function(.) round(seq(.[1],.[2],length.out=9))))
#
subjectsSVO<-(within(subset(subjects,Treat_Name==1),{
    otherPay2<-SpiteTableSVO[2,`chosen_option[16]`]
    otherPay4<-SpiteTableSVO[4,`chosen_option[19]`]
    otherPay6<-SpiteTableSVO[6,`chosen_option[17]`]
    otherPay8<-SpiteTableSVO[8,`chosen_option[20]`]
    otherPay10<-SpiteTableSVO[10,`chosen_option[18]`]
    otherPay12<-SpiteTableSVO[12,`chosen_option[21]`]
    }))
Personality<-join(Personality,subjectsSVO[,grep("^sid$|^otherPay[1-9]",names(subjectsSVO),value=TRUE)])
## Nameschosen_option<-grep("*option\\[[0-9]+\\]$",names(subjects),value=TRUE)
#
#200 is maximally possible to give to the other - every number lower shows how many tokens were taken away!
Personality<-within(Personality,{
    Spitefullness_inequality               <-200-otherPay2-otherPay4;
    Spitefullness_overall <-570-(otherPay2+otherPay4+otherPay6+otherPay8+otherPay10+otherPay12)
    Spitefullness_relativeGain             <-100-otherPay6
    Spitefullness_pure                     <- 85-otherPay10
    Spitefullness_inequality_willingtoPay  <-100-otherPay4
    Spitefullness_relativeGain_willingtoPay<-100-otherPay8
    Spitefullness_pure_willingtoPay        <- 85-otherPay12})
#
NsubWillingToPay<-sum(apply(Personality[,grep("Spitefullness_.*willingtoPay",names(Personality),value=TRUE)],1,sum)>0)
NsubWillingToPayPerCent<-round(mean(apply(Personality[,grep("Spitefullness_.*willingtoPay",names(Personality),value=TRUE)],1,sum)>0)*100)
##
NsubSpitefulPayPerCentAlt<-round(mean(Personality$Spitefullness_overall>0)*100)
#
NsubSpiteful<-sum(sbj[["Spitefullness_overall"]]!=0) ## <- is this correct?
NsubSpitefulPayPerCent<-round(NsubSpiteful/dim(sbj)[1],2)*100
#basically not much changes but the principal component analysis shows that 97\% can be explained by the first element for our measure.
#also using a binary version would not change much!
#Personality$Spitefullness_overall<-ifelse(Personality$Spitefullness_overall==0,0,1)
#adding the measures of the kimbourgh spite measure (time needed for testquestions+each stage 
#+approaches need to answer testquestions + the bid adaptations (losing and winning) 
#+ willingess to pay fo the bid adaptations (losing and winning))
#Approaches needed for anwesering the testquestions.
#
Personality<-join(Personality,subset(subjects,Treat_Name==2 & Period==ifelse(AllPay_First==0,1,ifelse(Date==1506170751,21,16)))[,grep("^sid$|[1-3_]Kim(|_fb)$|^Slider[12]|^Willingness_(loser|winner)$|counter_AllPay|Dauer_Testfrage_AllPay",names(subjects),value=TRUE)])
Personality<-rename(Personality,c("Slider1"="Slider_loser","Slider2"="Slider_winner"))
#Time needed for anwesering the testquestions: Dauer_Testfrage_Kim
#Note: In the first session we did not have time to run the testquestions - hence, they are NA
#Time needed for input of the bids: Dauer_Stage1Kim
#Time needed for understanding the feedback: Dauer_Stage1Kim_fb
#Time needed adapt the bids: Dauer_Stage2Kim
#Time needed for willingess to pay: Dauer_Stage3Kim
#Adapting the losing bid -basically spite: Slider_loser
#Adapting the winning bid: Slider_winner
#willingess to pay for adapting the winning bids: Willingness_winner
#willingess to pay for adapting the losing bids -basically willingness to pay for spite: Willingness_loser
#---------------------------------------------------------------
# The time and the amount of tries needed to anwsers the testquestions for the all-pay auction: counter_AllPay, Dauer_Testfrage_AllPay
#Note: After the first session we gave a hint to the testquestions. Therefore, the distribution of the first and the other seven session are different. In the first session people had a harder time to figure out the questions. In the remaining session most people anwsered pretty good (~86 anweser the questions correctly)

#this was the right way:
sbj<-join(sbj,Personality) #combining the subjects' demographics with the personality information


## ----DefinitionOfFunctions,error=FALSE,include=FALSE,warning=FALSE------------
##############
##Theoretical Bidding Function for Spitefull in this auction
##############
spite <- function(x,alpha){ 
  ((alpha+1)/(alpha*(2*alpha-1)))*((1-alpha)*((1-x)^(alpha/(1-alpha))-1)+alpha*x)
}
##############
##Theoretical Bidding Function for risk averse in this auction
##############
Risk <- function(x,r){ 
  integrand <- function(v) {r*(1-exp(-v/r))/(1-v)}
integrate(integrand, lower = 0, upper = x)$ value
}
## risk in the first-price auction
# note: Morgan et al (and others) derived the equilibrium behavior
# with CRRA risk preferences - using CARA does not lead to a 
# solution (at least I'm currently not able to find the analytic 
# solution) - also using a ode (see last chunck) does not lead to
# any solution!!
RiskFSP <- function(x,r){ x*(1/(1+r))}
##############
##Theoretical Bidding Function for Spitefull in the first-price auction
##############
spiteFSP <- function(x,alpha){ ((alpha+1)/(alpha+2))*x}
##############
##Theoretical Bidding Function for neutral player in this auction
##############
spiteless<-function(x){
  -x-log(1-x)
}
##############
##Theoretical Bidding Function for neutral player in the first-price auction
##############
spitelessFSP<-function(x){
  x/2
}
##############
##Player Type Function (individual bidding behavior)
##############
PlayerType<-function(InpMatrix){
      ###please insert a Matrix with only bids (Auction[,1:11])
    type_of_player<-matrix("hump_shaped",dim(InpMatrix)[1],1)
    for(i in 1:dim(InpMatrix)[1]){
      Array<-as.numeric(InpMatrix[i,])
    
      if(sum(diff(Array)>=0)==10){
          type_of_player[i]<-"increasing"
      }
      if(sum(diff(Array)==0)==10){
          type_of_player[i]<-"constant"
      }
      if(sum(Array)<100){
        type_of_player[i]<-"cheapWin"
      }
        if(sum(Array==0)==11){
        type_of_player[i]<-"zeros"
      }
      if(sum(Array==150)==11){
        type_of_player[i]<-"maxs"
      }
    }
  return(type_of_player)
}
##############
##Overbidding (i.e. actual bids minus theoretical bids)
##THIS TIME WE USE THE ACUTAL MAXIMUM OF 150 - THE MAXIMUM ALLOWED BID WAS 150 
##150 was the maximum to ensure that participants still got a postive payoff in the end
##############
Overbidding<-function(InpMatrix){
overbid<-data.frame(matrix(-1,dim(InpMatrix)[1],11))
for(i in 1:dim(InpMatrix)[1]){
  overbid[i,]<-as.vector(InpMatrix[i,]-c(spiteless(0),spiteless(10/100)*100,spiteless(20/100)*100,
                               spiteless(30/100)*100,spiteless(40/100)*100,spiteless(50/100)*100,
                               spiteless(60/100)*100,spiteless(70/100)*100,spiteless(80/100)*100,
                               spiteless(90/100)*100,150))
}
  return(overbid)
}
##############
##Overbidding in the first-price auction
##############
OverbiddingFSP<-function(InpMatrix){
overbid<-data.frame(matrix(-1,dim(InpMatrix)[1],11))
for(i in 1:dim(InpMatrix)[1]){
  overbid[i,]<-as.vector(InpMatrix[i,]-c(spitelessFSP(0),spitelessFSP(10/100)*100,spitelessFSP(20/100)*100,
                               spitelessFSP(30/100)*100,spitelessFSP(40/100)*100,spitelessFSP(50/100)*100,
                               spitelessFSP(60/100)*100,spitelessFSP(70/100)*100,spitelessFSP(80/100)*100,
                               spitelessFSP(90/100)*100,spitelessFSP(1)*100))
}
  return(overbid)
}


## ----DataPreparationForRegressions,error=FALSE,include=FALSE,warning=FALSE----
##############
##Melting the data such that each choice is in each row -for the general auction data
##############
aa<-join(Auction,rename(with(Auction,aggregate(Period~Date,FUN=min)),c("Period"="MinPeriod")))
Auction<-within(aa,Period<-Period-MinPeriod+1)
Auction_melt <- melt(Auction, id=grep("Choice",names(Auction),invert=TRUE,value=TRUE))
#Order such that, we have each decision of each period of each participant
Auction_melt<-Auction_melt[with(Auction_melt,order(Date,Subject,Period)),]
##############
##Preparing the data such, that we can make regressions!
##############
daten<-join(Auction_melt,sbj[,c("sid","Spitefullness_overall","Spite_Quest","Geschlecht","Slider_loser","Risk_overall","Alter","rivalry","svo_type","SVO","inequality_aversion_score","Spitefullness_inequality")])

daten<-rename(daten,c("Spitefullness_overall"="Spite_SVO"))
daten<-rename(daten,c("Slider_loser"="Spite_Kim"))
daten<-rename(daten,c("inequality_aversion_score"="IAS"))
daten<-rename(daten,c("Spitefullness_inequality"="IA"))

daten$variable<-as.numeric(sub("Choice","",daten$variable))*10-10
daten$SpiteSum[daten$FirstPrice==0]<- as.numeric(scale(daten$Spite_Quest[daten$FirstPrice==0]) +  scale(daten$Spite_SVO[daten$FirstPrice==0])+ scale(daten$Spite_Kim[daten$FirstPrice==0]))
daten$SpiteSum[daten$FirstPrice==1]<- as.numeric(scale(daten$Spite_Quest[daten$FirstPrice==1]) +  scale(daten$Spite_SVO[daten$FirstPrice==1])+ scale(daten$Spite_Kim[daten$FirstPrice==1]))

daten$spiteless[daten$FirstPrice==0]<-ifelse(daten$variable[daten$FirstPrice==0]<100,100*spiteless(daten$variable[daten$FirstPrice==0]/100),150)
daten$spiteless[daten$FirstPrice==1]<-spitelessFSP(daten$variable[daten$FirstPrice==1]/100)*100

daten$overbid <- daten$value-daten$spiteless
## ----------------------------------------------------------------------------------------------------
## ----------------             Overbidding                 -------------------------------------------
## ----------------------------------------------------------------------------------------------------
##############
##Building the overbidding data out of the auction data
## NOTE: WITH THE DATA WHICH USES 150 AS MAXIMUM
##############
overbid1<-Overbidding(Auction[Auction$FirstPrice==0,grep("^Choice[0-9]+$",names(Auction))])
overbid11<-cbind(overbid1[,1:11],Auction[Auction$FirstPrice==0,c("Date","Subject","Period","gid","sid")])
overbid2<-OverbiddingFSP(Auction[Auction$FirstPrice==1,grep("^Choice[0-9]+$",names(Auction))])
overbid22<-cbind(overbid2[,1:11],Auction[Auction$FirstPrice==1,c("Date","Subject","Period","gid","sid")])
overbid<-rbind(overbid11,overbid22)
names(overbid)<-sub("^X","Choice",names(overbid))
##############
##Melting the data such that each choice is in each row -with the overbidding data
## NOTE: WITH THE DATA WHICH USES 150 AS MAXIMUM
##############
overbid_melt <- melt(overbid, id=colnames(overbid[,-grep("^Choice",names(overbid))]))
#Order such that, we have each decision of each period of each participant
str(overbid_melt)
overbid_melt<-overbid_melt[with(overbid_melt,order(sid,Period,variable)),]
##############
##Preparing the data such, that we can make regressions! For the overbidding Data
## NOTE: WITH THE DATA WHICH USES 150 AS MAXIMUM
##############
daten_overbid<-join(overbid_melt,sbj[,c("sid","Spitefullness_overall","Spite_Quest","Geschlecht","Slider_loser","Risk_overall","Alter","rivalry","svo_type","SVO","inequality_aversion_score","Spitefullness_inequality","FirstPrice")])
daten_overbid<-rename(daten_overbid,c("Spitefullness_overall"="Spite_SVO"))
daten_overbid<-rename(daten_overbid,c("Slider_loser"="Spite_Kim"))
daten_overbid<-rename(daten_overbid,c("inequality_aversion_score"="IAS"))
daten_overbid<-rename(daten_overbid,c("Spitefullness_inequality"="IA"))
daten_overbid$variable<-as.numeric(sub("Choice","",daten_overbid$variable))*10-10
#daten_overbid$SpiteSum<-as.numeric(scale(daten_overbid$Spite_Quest) +  scale(daten_overbid$Spite_SVO)+ scale(daten_overbid$Spite_Kim))
#to get the same results as before we need to make the scale relative to the treatments (maybe spite has changed or so...)
daten_overbid$SpiteSum[daten_overbid$FirstPrice==0]<-as.numeric(scale(daten_overbid$Spite_Quest[daten_overbid$FirstPrice==0]) +  scale(daten_overbid$Spite_SVO[daten_overbid$FirstPrice==0])+ scale(daten_overbid$Spite_Kim[daten_overbid$FirstPrice==0]))
daten_overbid$SpiteSum[daten_overbid$FirstPrice==1]<-as.numeric(scale(daten_overbid$Spite_Quest[daten_overbid$FirstPrice==1]) +  scale(daten_overbid$Spite_SVO[daten_overbid$FirstPrice==1])+ scale(daten_overbid$Spite_Kim[daten_overbid$FirstPrice==1]))


## ----PValFunc,error=FALSE,include=FALSE,warning=FALSE-------------------------
PVal<-function(Value){
    if(is.na(Value))
        return(NA)
    Result<-sprintf("%.5f",Value)
    if (as.numeric(Result)==0)
        Result<-"< .00001"
  return(Result)
}


## ----ModelCompReport,error=FALSE,include=FALSE,warning=FALSE------------------
ModCompReport<-function(Test){
  df<-abs(Test$Df[2]-Test$Df[1])
  t<-round(as.numeric(Test$`Chisq diff`[2]),3)
  p<-PVal(Test$`Pr(>Chisq)`[2])
  Result<-paste("$\\chi^2_{",df,"}= ",t,"$, $p",p,"$",sep="")
  return(Result)
}
## 
ModCompReportLR<-function(Test,Risk=FALSE){
  df<-Test$Df[2]
  t<-round(as.numeric(Test$Chisq[2]),3)
  p<-PVal(Test$`Pr(>Chisq)`[2])
  Result<-paste("$\\chi^2_{",df,"}= ",t,"$, $p",p,"$",sep="")
  if(Risk){
    Result<-paste("$\\chi^2_{",df,"}= ",t,"$",sep="")
  }
  return(Result)
} 


## ----CorrelationReport,error=FALSE,include=FALSE,warning=FALSE----------------
CorrReport<-function(Test,Index1=1,Index2=1){
  r<-round(as.numeric(Test$r[Index1,Index2]),3)
  p<-PVal(Test$p[Index1,Index2])
  Result<-paste("$r= ",r,"$, $p=",p,"$",sep="")
  return(Result)
}


## ----FunctionForPlot,error=FALSE,include=FALSE,warning=FALSE------------------
## Gives count, mean, standard deviation, standard error of the mean, and confidence interval (default 95%).
##   data: a data frame.
##   measurevar: the name of a column that contains the variable to be summariezed
##   groupvars: a vector containing names of columns that contain grouping variables
##   na.rm: a boolean that indicates whether to ignore NA's
##   conf.interval: the percent range of the confidence interval (default is 95%)
summarySE <- function(data=NULL, measurevar, groupvars=NULL, na.rm=FALSE,
                      conf.interval=.95, .drop=TRUE) {
  # New version of length which can handle NA's: if na.rm==T, don't count them
  length2 <- function (x, na.rm=FALSE) {
    if (na.rm) sum(!is.na(x))
    else       length(x)
  }
  # This does the summary. For each group's data frame, return a vector with
  # N, mean, and sd
  datac <- ddply(data, groupvars, .drop=.drop,
                 .fun = function(xx, col) {
                   c(N    = length2(xx[[col]], na.rm=na.rm),
                     mean = mean   (xx[[col]], na.rm=na.rm),
                     sd   = sd     (xx[[col]], na.rm=na.rm)
                   )
                 },
                 measurevar
  )
  
  # Rename the "mean" column    
  datac <- plyr::rename(datac, c("mean" = measurevar))
  datac$se <- datac$sd / sqrt(datac$N)  # Calculate standard error of the mean
  # Confidence interval multiplier for standard error
  # Calculate t-statistic for confidence interval: 
  # e.g., if conf.interval is .95, use .975 (above/below), and use df=N-1
  ciMult <- qt(conf.interval/2 + .5, datac$N-1)
  datac$ci <- datac$se * ciMult
  return(datac)
}
##
SummaryStat<-function(daten,alpha){
  data.summary <- data.frame(
    treatment=c(3,12),
    mean=tapply(daten$AbuseEver, daten$Power, mean),
    n=tapply(daten$AbuseEver, daten$Power,  length),
    sd=tapply(daten$AbuseEver, daten$Power,  sd)
  )
  # Precalculate standard error of the mean (SEM)
  data.summary$sem <- data.summary$sd/sqrt(data.summary$n)
  # Precalculate margin of error for confidence interval
  data.summary$me <- qt(1-alpha/2, df=data.summary$n)*data.summary$sem
  return(data.summary)
}


## ----GamToTabelFunc,error=FALSE,include=FALSE,warning=FALSE-------------------
stargazerGamPrep<-function(Object,mycoeffnames=NA,k=2){
A<-summary(Object$gam)  
N<-length(A$p.coeff)
if(is.na(mycoeffnames)){
  GivenNames<-names(A$p.coeff)
}else{
  GivenNames<-mycoeffnames
}
##
Result<-as.data.frame(matrix( 0,N+3,1))
 B<-as.data.frame(matrix(0,N,1))
 for(i in 1:N){
   star<-symnum(A$p.pv[i], cutpoints = c(0.001, 0.01, 0.05, 0.1,1,0),
                symbols = c("$^{***}$", "$^{**}$", "$^{*}$", "$^{+}$",""))
                #symbols = c("***", "**", "*", "+",""))
   B[i,]<-paste(ifelse(abs(round(A$p.coeff[i],k))<0.001,ifelse(round(A$p.coeff[i],k)<0,-0.001,.001),sprintf("%3.2f",round(A$p.coeff[i],k))),
                star," (",ifelse(round(A$se[i],k)<0.001,0.001,sprintf("%3.2f",round(A$se[i],k))),")",sep="")
 }
 Result<-rbind(B,A$n,round(as.numeric(logLik(Object$lme)),k),round(AIC(Object$lme),k),round(BIC(Object$lme),k))
 rownames(Result)<-c(GivenNames,"Observations","Log Likelihood","Akaike Inf. Crit","Bayesian Inf. Crit.")
 return(Result)
}
##
stargazerGam<-function(Objects,multipleObjects=FALSE,mycoeffnames=NA,renamed=NA,Order=NA,k=3,ModelNames=NA){
  base<-as.data.frame(matrix(0,length(mycoeffnames),1))
  rownames(base)<-mycoeffnames
  base$rn <- rownames(base)
  Result<-base
  if(multipleObjects){
    for(i in 1:length(Objects)){
      A<-stargazerGamPrep(Objects[[i]])
      A$rn <- rownames(A)
      tmp<-join_all(list(Result,A), by = 'rn', type = 'left')#merge(Result,A,by="row.names", all=TRUE)
      Result<-tmp
    }
  }else{
    A<-stargazerGamPrep(Objects)
    A$rn <- rownames(A)
    tmp<-join_all(list(Result,A), by = 'rn', type = 'left')#merge(Result,A,by="row.names", all=TRUE)
    Result<-tmp
  }
  #drop the info of the base

  if(!is.na(renamed)[1]){
    rownames(Result)<-renamed
  }else{
    rownames(Result)<-Result[,2]
  }
  Result<-Result[,-c(1,2)]
  if(!is.na(ModelNames)[1]){
    names(Result)<-ModelNames
  }
  Result[is.na(Result)]<-""
  return(Result)
}
##
PrintstargazerGam<-function(Objects,multipleObjects=FALSE,mycoeffnames=NA,renamed=NA,Order=NA,k=3,ModelNames=NA,newnames=NA){
  
  n<-length(mycoeffnames)
  n1<-length(ModelNames)
  #myalign<-paste(c("@{\\extracolsep{-5pt}}","l",rep("c",n1)),sep="",collapse = "")
  myalign<-paste(c("l",rep("c",n1)),sep="",collapse = "")
  
  GamTable<-stargazerGam(Objects,
             multipleObjects=multipleObjects,
             mycoeffnames=mycoeffnames,renamed=renamed,Order=Order,k=k,
             ModelNames=ModelNames)

  if(!(is.na(newnames)[1])){
    m<-length(newnames)
    rownames(GamTable)[1:m]<-newnames
  }

print(xtable(GamTable, align=myalign),hline.after=NULL, #booktabs=TRUE,
      add.to.row = list(list(-1,0,n),
                        c("\\toprule \n","\\midrule \n",
                        paste("\\bottomrule \n %\\\\ 
\\multicolumn{",n1+1,"}{l}{\\textit{Notes:} \\pStarNotes}",sep=""))))
#                        $^{+}$p $<$ 0.1;$^{*}$p $<$ 0.05;$^{**}$p 
#                        $<$ 0.01;$^{***}$p $<$ 0.001;}",sep=""))))
}


## ----include=FALSE,cache=FALSE------------------------------------------------
optAR<-list(pval=c(NA,NA),citext=c(NA,NA))
if(file.exists("reviewer.Rdata"))
    load("reviewer.Rdata")


## ----example2ndprice,fig.width=6.5--------------------------------------------
p12<-function (a1,v1,a2,v2,x1=c(0,1.15),x2=c(0,1.15),upper=1,ext=1.5,parName="\\alpha",keepT=seq(1,by=1,length.out=length(v2))) {
    #v1=seq(0,1,.012499);a1=c(-.49999,0.00001,.25,.500001,.75,1);v2=seq(.0001,1,.0999);a2=seq(0.0001,1,.02499)
    #x1=c(0,1.15);x2=c(0,1.15);upper=1;ext=1.5;parName="\\alpha"
    xx1<-within(ldply(a1,function(a) data.frame(list(v=v1,b=sapply(v1,bidfun,a=a),a=a))),a<-round(a,2))
    y1<-extendrange(c(0,quantile(xx1$b,c(upper),na.rm=TRUE)),f=.1)
    xx1t<<-subset(xx1,v+b<ext)
    xx1t<<-within(join(with(xx1t,aggregate(v~a,FUN=max)),xx1t),{l<-paste("$",ifelse(a==max(a1),paste(parName,"="),""),round(a,2),"$");pos<-ifelse(v>.99,4,ifelse(a==max(a1),2,0))})
    p1<<-xyplot(b~v,group=a,data=xx1,type="l",xlab="$v$",ylab="$b$",ylim=y1,xlim=x1)+layer(panel.text(xx1t$v,xx1t$b,label=xx1t$l,cex=.5,pos=xx1t$pos))
    ##
    xx2<<-within(ldply(v2, function(v) data.frame(list(v=v,b=sapply(a2,bidfun,v=v),a=a2))),v<-round(v,2))
    xx2t<<-within(subset(xx2,a==max(a)),{
        v<-paste("$",ifelse(v==round(max(v2),2),"v=",""),v,"$")
        if(parName=="r") a<-log10(a)
    })[keepT,]
    p2<<-xyplot(b~a,group=v,data=xx2,type="l",xlim=x2,ylim=y1,xlab=paste("$",parName,"$"),ylab="$b$",scales=list(x=list(log=parName=="r")),xscale.components=ifelse(parName=="r",xscale.components.log10.3,xscale.components.default),panel=function(...) {
    panel.xyplot(...)
    panel.text(xx2t$a,xx2t$b,label=xx2t$v,pos=4,cex=.5)})
    ##
    print(p1, split=c(1,1,2,1), more=TRUE)
    print(p2, split=c(2,1,2,1))
}
##
bidfun<-function(v,a) (a+1)/(a*(2*a-1))*((1-a)*((1-v)^(a/(1-a))-1)+v*a)
p12(v1=seq(0,1,.012499),a1=c(-.49999,0.00001,.25,.500001,.75,1),v2=seq(.0001,1,.0999),a2=seq(0.0001,1,.02499),upper=.95,ext=1.5,x2=c(0,1.25),keepT=seq(1,11,1))

## -----------------------------------------------------------------------------



## ----example2ndpriceRisk,fig.width=6.5----------------------------------------
bidfun<-function(v,a) {integrand <- function(x) {a*(1-exp(-x/a))/(1-x)}
   integrate(integrand, lower = 0, upper = v)$ value}
p12 (v1=seq(0.05,.95,.05),a1=c(0.1,.25,.5,1,30),x2=c(.01,100),a2=10^seq(-2,1.5,.2),v2=c(.0001,seq(.1,.9,.1)),upper=.977,ext=2,parName="r",keepT=c(1,4,6:11))


## ----example2ndpriceRiskAndSpite----------------------------------------------
bidfunComplex<-function(v,a,r) {
  integrand <- function(x) {((-exp(a*x/r)+exp(a/r))^(1/(a-1)))/(exp(x/r))}
  Part1<-integrate(integrand, lower = 0, upper = v,rel.tol = .Machine$double.eps^0.5)$ value
  Prep1<-((a-1))
  LogTop<- r*(a-1)*(exp(a/r)-1)^(-a/(a-1))
  LogButton<-((a^2)*Part1*(exp(a/r)-1)^(-a/(a-1))-(r*(a-1)))
  Add<-a*log(-exp((a*v/r))+exp((a/r)))
  Divide<-((a-1)*a)
  Full<- r*(Prep1*log(-LogTop/LogButton)+Add)/Divide
  return(Full-spiteless(v)
  )
}

bidfun<-function(v,a,r) {bidfunComplex(v,a,r)}


bid2df<-function(a=seq(.1,.9,length.out=9),v1=(1-seq(1,0.03,length.out=100)),r=.5,hide=c(),bmin=-.7) {
    overV <- length(v1)>length(a)
    aName<-ifelse(overV,"\\alpha","v")
    vName<-ifelse(overV,"v","\\alpha")
    ##
    n<-max(length(a),length(r))
    a<-rep_len(a,length.out=n)
    r<-rep_len(r,length.out=n)
    bids<-ldply(1:n,function(i) data.frame(a=a[i],r=r[i],v=v1,b=sapply(v1,bidfun,a=a[i],r=r[i])))
    bids<-subset(bids,b>bmin)
    if (!overV)
        names(bids)<-c("v","r","a","b")
    bids[["distance"]]<-sapply(1:nrow(bids),function(i) with(subset(bids,a != bids[i,"a"] | r != bids[i,"r"]),min((v-bids[i,"v"])^2+(b-bids[i,"b"])^2)))
    bids<-within(bids,distance[v>.9 | v<.1]<-0) ## labels should not go too far to the right
    bids<-ddply(bids,.(a,r),mutate,dotext=(max(distance)==distance))
    bids<-within(bids,{
        if(sd(a)>0) ltext<-sprintf("$%s=%.2g$",aName,a)
        if(sd(r)>0) ltext<-sprintf("$r=%.2g$",r)
        if(sd(r)>0 & sd(a)>0) ltext<-sprintf("$%s=%.2g,r=%.2g$",aName,a,r)
        sltext<-ifelse(dotext,ltext,"")
    })
    tt<-subset(bids,dotext)
    if(length(hide)>0)
        tt<-tt[-hide,]
    striptext<-""
    if(sd(bids[["a"]])==0)
        striptext<-sprintf("$%s=%.2g$",aName,mean(bids[["a"]]))
    if(sd(bids[["r"]])==0)
        striptext<-sprintf("%s%s$r=%.2g$",striptext,ifelse(striptext!="",",",""),mean(bids[["r"]]))
    xyplot(b~v | factor(striptext),groups=ltext,data=bids,type="l",xlab=paste0("$",vName,"$"),ylab="$b-b^\\IIAP$",strip=striptext!="",
           par.strip.text=list(cex=.7),
           panel=function(...) {panel.xyplot(...);panel.text(tt[["v"]],tt[["b"]],tt[["ltext"]],cex=.4)})
}

## ----example2ndpriceRiskAndSpite2,fig.width=6.5-------------------------------
## p1<-bid2df(a=.2,r=2^(seq(-2,2,length.out=9)),hide=c(2,4,8))
## p2<-bid2df(a=.4,r=2^(seq(-2,2,length.out=9)),hide=c(2,6,8))
## p3<-bid2df(a=.6,r=2^(seq(-2,2,length.out=9)),hide=7:8)
## c(p1,p2,p3,y.same=TRUE)
c(bid2df(r=.25,hide=c(2:5)),
  bid2df(r=.5,hide=c(2:3)),
  bid2df(r=1),y.same=TRUE)


## ----graphsDistributionSpite1a, cache=TRUE, cache.comments=FALSE, fig.height=2.5, fig.width=6----
xx<-melt(subset(sbj,FirstPrice==0)[,c("Spite_Quest","Slider_loser","Spitefullness_overall","Geschlecht")])
#xx<-melt(sbj[,c("Spite_Quest","Slider_loser","Spitefullness_overall","Geschlecht")])
xx$variable<-revalue(xx$variable,c("Spite_Quest"="Marcus et al.","Slider_loser"="Kimbrough-Reiss","Spitefullness_overall"="Own measure"))
ecdfplot(~value | variable,data=xx,scale="free",xlab="Measures for Spite",
            plot.points=FALSE,auto.key = list(corner = c(0.95, 0.1),cex=.7))


## ----ownSpiteMeasure----------------------------------------------------------
ownMeasure<-rbind.fill(alply(read.table(header=TRUE,text='
IA1 IA2 RG1 RG2 PS1 PS2
 70  70 100 100 100 100 
 70  50 100  85 100  85
100 100 100 100  85  85
 85  70  85  70  70  50
'),2,function(.) 
    data.frame(list(name=names(.),
      you=round(seq(.[1,],.[2,],length.out=9)),
    other=round(seq(.[3,],.[4,],length.out=9))))))
ownTxt<-aggregate(ownMeasure[,c("you","other")],list(name=ownMeasure$name),FUN=mean)


## ----ownMeasureTab1,results='asis'--------------------------------------------
q<-by(ownMeasure,list(ownMeasure[["name"]]),function(x) {
    q<-cat(paste("\\AA{",x[1,"name"],"}",sep=""))
    sapply(1:nrow(x),function(i) q<-cat(sprintf("\\BB{%d}{%d}",x[i,"you"],x[i,"other"])))
    cat("\\CC\\\\\n")
})


## ----ownMeasurePlot,fig.width=3.5---------------------------------------------
xyplot(other ~ you,group=name,par.settings=list(par.xlab.text=list(cex=.8),par.ylab.text=list(cex=.8),add.text=list(cex=.7)),data=ownMeasure,type=c("p","l"),panel=function(...) {
    panel.xyplot(...)
    panel.text(ownTxt$you,ownTxt$other,ownTxt$name,pos=ifelse(substr(ownTxt$name,3,3)=="1",4,2))
    panel.abline(a=0,b=1)
},xlim=c(48,109),xlab="Own payoff",ylab="Other payoff")


## ----mgTab,results='asis'-----------------------------------------------------
mgTab<-with(aggregate(sid ~ gid,FUN=function(x)
    length(unique(x)),data=subset(subjects,FirstPrice==0)),table(sid))


## ----zTree.data,results="asis"------------------------------------------------
dd<-data.frame(list(x=seq(0,100,10),
                    y=c(0,1,3,5,10,14,18,30,42,65,100)))
fbEx<-data.frame(list(auctions=c(9,7,2,4,1,3,8,10,6,5),
                    valuationsReal=c(20,31,34,40,42,45,58,72,84,100),
                    bidsReal=c(3,6,7,10,11,12,17,32,51,100),
                    valuationsOther=rep("?",10),
                    bidsOther=c(28,13,4,4,11,13,2,3,1,1)))
fbEx<-within(fbEx,{Result<-ifelse(bidsReal>=bidsOther,"Won","Lost");
                    Points<-ifelse(Result=="Lost",-bidsReal,valuationsReal-bidsOther)
                    })


## ----zTree.input,results='asis'-----------------------------------------------
with(dd,print(xtable(rbind(`Your valuation`=x,
                           `Your bid`=sprintf("\\INPUTOLIVER{%d}",y)),align=rep('c',length(x)+1)),
      include.colnames=FALSE,hline.after=NULL))


## ----zTree.plot.input,fig.height=3--------------------------------------------
par(mar=c(3,4,.5,.5))
plot(y~x,ylim=c(0,150),data=dd,t="l",las=1,xaxs='i',yaxs='i',xaxp=c(0,100,10),yaxp=c(0,150,15),xlab="Valuation",ylab="Bid",col="blue",lwd=3)
abline(v=seq(0,100,10));abline(h=seq(0,150,10))


## ----prepareZtreePlotFeedback,results='asis'----------------------------------
with(fbEx,
     print(xtable(rbind(`Auction`=auctions,
           `Your valuation`=valuationsReal,
           `Your bid`=bidsReal,
           `Other's valuation`=as.character(valuationsOther),
           `Other's bid`=bidsOther,
           `Won/lost`=ifelse(Result=="Lost","\\color{red} lost","\\color{green} won"),
           `Points gained/lost`=Points),align=c('|l|',rep('c|',length(auctions)))),
      include.colnames=FALSE,hline.after=0:7))


## ----zTree.plot.feedback------------------------------------------------------
par(mar=c(4,4,.5,4))
plot(y~x,ylim=c(0,150),xaxs='i',yaxs='i',data=dd,t="l",las=1,xaxp=c(0,100,10),yaxp=c(0,150,15),xlab="Your valuation",ylab="Bid",bty='n',lwd=3)
abline(v=seq(0,100,10));
abline(h=seq(0,150,10));
par(xpd=NA)
with(fbEx,{
points(valuationsReal,bidsOther,col="blue",pch=4,lwd=2)
points(valuationsReal,bidsReal,col=ifelse(Result=='Lost',"red",'green'),pch=5,lwd=2)
text(max(valuationsReal),max(bidsReal),"Your bid",pos=4)
text(max(valuationsReal),0,"Other bid",pos=4,col='blue')
text(valuationsReal,bidsReal,auctions,adj=c(-.5,1),cex=.5)
})
legend(x=-25,y=-15,bg="white",c("Bids in auctions lost","Bids in auctions won"),pch=5,col=c("red","green"),bty="n")


## ----fig.Joint.Spite,fig.width=6,fig.height=2.5-------------------------------
parS<-list(layout.widths=list(left.padding=0,right.padding=0,ylab.axis.padding=0,axis.key.padding=0))
plot(xyplot(jitter(Slider_loser)~jitter(Spite_Quest),data=sbj,xlab="Marcus et al.",ylab="Kimbrough-Reiss",par.settings=parS),split=c(1,1,3,1),more=TRUE)
plot(xyplot(jitter(Slider_loser)~jitter(Spitefullness_overall),data=sbj,xlab="Own measure",ylab="Kimbrough-Reiss",par.settings=parS),split=c(2,1,3,1),more=TRUE)
plot(xyplot(jitter(Spite_Quest)~jitter(Spitefullness_overall),data=sbj,xlab="Own measure",ylab="Marcus et al.",par.settings=parS),split=c(3,1,3,1))


## ----cronbachAlpha------------------------------------------------------------
fcv<-function(data,i) {
    pc<-princomp(data[i,])
    c(pc1=100*(pc$sdev^2)[1]/sum(pc$sdev^2),ca=cronbach(data[i,])$alpha)
    }
boot2ci <- function(boot,index=1) paste(signif(boot$t0[index],3),ifelse(index==1,"\\% of the variance,","")," ($\\text{CI}=[",paste(signif(boot.ci(boot,type="norm",index=index)$normal[-1],3),collapse=","),"]$)",sep="")
#
set.seed(123)
#
spQuestNames<-grep("^Spite[0-9]+$",names(sbj),value=TRUE)
spQuestData<-sapply(spQuestNames,function(n) as.numeric(sbj[,n])) %*% diag(c(spiteQuestVal))
Quest.boot<-boot(spQuestData,fcv,R=1000)
#
spOwnNames<-grep("Spitefullness_[irp]",names(sbj),value=TRUE)
spOwnData<-sbj[,spOwnNames]
Own.boot<-boot(spOwnData,fcv,R=1000)

allSpiteNames<-c("Spite_Quest","Slider_loser","Spitefullness_overall")
allSpiteData<-sbj[,allSpiteNames]
allSpite.boot<-boot(allSpiteData,fcv,R=1000)

CorrelationTest<-corr.test(allSpiteData)
sbj2<-merge(sbj[,c("SVO","sid")],unique(daten[,c("SpiteSum","sid")]),by="sid")
CorrelationTestSVO<-corr.test(sbj2[,c("SVO","SpiteSum")])


## ----fig.distr.Risk,fig.width=3.5, fig.height=2.6-----------------------------
rnChoice <- sum(sapply(1:10,function(i) i*1800+(10-i)*1440 > i*3465 + (10-i)*90))
riskCat<-cut(sbj[["Risk_overall"]],c(-1,rnChoice-.5,rnChoice+.5,11),labels=c("loving","neutral","averse"))
riskTab<-round(prop.table(table(riskCat))*100,2)
##
histogram(sbj$Risk_overall,xlab="Holt and Laury measure for risk attitude",freq=FALSE, main="",breaks=((0:11)-.5)) +
layer(panel.curve(100*dnorm(x, mean=mean(sbj$Risk_overall), sd=sd(sbj$Risk_overall)), add=TRUE, col="darkblue", lwd=2),
      panel.abline(v=4,lwd=2),panel.text(4,riskTab["neutral"],"risk neutral",cex=.8,srt=90,adj=c(-0.1,-.40)))
##
sbj3<-merge(sbj[,c("SVO","Risk_overall","sid")],unique(daten[,c("SpiteSum","sid")]),by="sid")
CorrelationTestRisk<-corr.test(sbj3[,c("SVO","SpiteSum","Risk_overall")])


## ----DataForTest,include=FALSE------------------------------------------------
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Data Preparation Secondprice all-pay - scaling 
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
data_for_test<-daten[daten$FirstPrice==0,]
data_for_test[,c("Spite_SVO","Spite_Quest","Spite_Kim","Risk_overall","rivalry","svo_type","IA","IAS")] <-
  scale(daten[daten$FirstPrice==0,c("Spite_SVO","Spite_Quest","Spite_Kim","Risk_overall","rivalry","svo_type","IA","IAS")])
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Data Preparation Firstprice winner-pay - scaling 
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
data_for_testFSP<-daten[daten$FirstPrice==1,]
data_for_testFSP[,c("Spite_SVO","Spite_Quest","Spite_Kim","Risk_overall","rivalry","svo_type","IA","IAS")] <-
  scale(daten[daten$FirstPrice==1,c("Spite_SVO","Spite_Quest","Spite_Kim","Risk_overall","rivalry","svo_type","IA","IAS")])


## ----WhichTheoryIsBetter,include=FALSE----------------------------------------
##here I basically fit the bidding behavior with the theory and show that theory on risk is suggesting a risk aversion paramter close to risk neutrality and that spite suggestes a spite-factor of .75.
##theory of spite explains data better then theory on risk!
##I offset the betas so that we can compare purely theory!
## !!! I think that we are using “normalised” values below, so we need normalised bids here,too !!!
data_for_test$value2<-data_for_test$value/100
##
bidfun_Spite<-function(v,a){
    ##if(v==1){v<-.9999999999}#it can lead to issues for alpha>1 and v=1
    v<-pmin(.9999999999,v) 
    result<- (a+1)/(a*(2*a-1))*((1-a)*((1-v)^(a/(1-a))-1)+v*a)
    ##result2<-ifelse(result>1.5,1.5,result)
    return(pmax(0,pmin(1.5,result,1.5)))}
    ##return(ifelse(result2<0,0,result2))}
##  
spiteless_Real<-function(x){ pmin(1.5, -x-log(1-x)) } ##; return(ifelse(result>1.5,1.5,result))}
bidfun_Risk<-function(v,a) { ##if(v==1){v<-.9999999999}
    v<-pmin(.9999999999,v)
    integrand <- function(x) {a*(1-exp(-x/a))/(1-x)}
    pmin(1.5,integrate(integrand, lower = 0, upper = v)$value)} ##; return(ifelse(result>1.5,1.5,result))}
##  
Overbid_ModelFit_Spite <- function(Alpha,ShowResult=0){
    if(Alpha==0.5){Alpha<-.4999999999}
    if(Alpha==0){
      ModelBid1_1<-lmer(value2 ~ offset(spiteless_Real(variable/100)) -1 +(1|sid) +(1|gid),data=data_for_test)
    }else{
      ModelBid1_1<-lmer(value2 ~ offset(bidfun_Spite(variable/100,Alpha)) -1+(1|sid) +(1|gid),data=data_for_test)
    }
    
    -as.numeric(summary(ModelBid1_1)$logLik)
    if(ShowResult==1){
      return(ModelBid1_1)
    }else{
      return(-as.numeric(summary(ModelBid1_1)$logLik))
    }
  }
##  
Overbid_ModelFit_Risk <- function(R,ShowResult=0){
    variable<-with(data_for_test,unique(variable))
    data_for_test<-join(data_for_test,data.frame(variable=variable,Theory=mapply(bidfun_Risk,variable/100,MoreArgs=list(a=R)))) ##  ← this is faster, equilibrium bids are calculated only once
    #data_for_test$Theory<-simplify2array(lapply(data_for_test$variable/100,bidfun_Risk,a=R)) ## ← this is correct, only slow
    ModelBid1_1<-lmer(value2 ~ offset(Theory) -1+(1|sid) +(1|gid),data=data_for_test)
    
    -as.numeric(summary(ModelBid1_1)$logLik)
    if(ShowResult==1){
      return(ModelBid1_1)
    }else{
      return(-as.numeric(summary(ModelBid1_1)$logLik))
    }
  }


## ----WhichTheoryIsBetterCombinedModel,include=FALSE---------------------------
#here I basically fit the bidding behavior with the theory and show that theory on risk is suggesting a risk aversion paramter close to risk neutrality and that spite suggestes a spite-factor of .75.
#theory of spite explains data better then theory on risk!
#I offset the betas so that we can compare purely theory!

bidfunComplex<-function(v,a,r) {
    integrand <- function(x) {pmin(1e99,((-exp(a*x/r)+exp(a/r))^(1/(a-1)))/(exp(x/r)))}
    v<-pmin(.999,v)
  Part1<-integrate(integrand, lower = 0, upper = v, stop.on.error=FALSE)$value ## values can be huge
  Prep1<-((a-1))
  LogTop<- r*(a-1)*(exp(a/r)-1)^(-a/(a-1))
  LogButton<-((a^2)*Part1*(exp(a/r)-1)^(-a/(a-1))-(r*(a-1)))
  Add<-a*log(-exp((a*v/r))+exp((a/r)))
  Divide<-((a-1)*a)
  Full<- r*(Prep1*log(-LogTop/LogButton)+Add)/Divide
  return(pmin(1.5,Full)) ## limit bids in the same way as above
}

data_for_Estimation<-data_for_test
data_for_Estimation$variableNew<-data_for_Estimation$variable/100
data_for_Estimation$variableNew[data_for_Estimation$variableNew==1]<-.999
data_for_Estimation$value2<-data_for_Estimation$value/100

#data_for_Estimation$Theory<-bidfunComplex(data$variableNew,a=Alpha,r=R))

#library(bbmle)
#R_Alpha_ML <- mle2(Overbid_ModelFit_SpiteAndRisk, start = list(Alpha=0.6, R = 10000000), method = "L-BFGS-B", lower = c(0.0001,0.0001), upper = c(.99,10000000),trace=TRUE)
#Note: The model gets stuck at a local optimum and never reaches the good solution. The best model is Overbid_ModelFit_SpiteAndRisk(.66,10000000) with a fit of 122675.8
# which is much better than what the model maximizer finds
##--
#a<-seq(0.001,.9,length.out=20)
#zaa<-mcmapply(Overbid_ModelFit_Spite,a)
#za<-mcmapply(Overbid_ModelFit_SpiteAndRisk,a,MoreArgs=list(R=10000))
#xyplot(zaa~za)
#a<-.6
#optimise(Overbid_ModelFit_Spite,c(0,1))
##--
#risk<-seq(.1,20,length.out=20)
#zr<-mcmapply(Overbid_ModelFit_Risk,risk)
#plot(zr~risk)
#optimise(Overbid_ModelFit_Risk,c(0,10000))
##--
##
##


## ----WhichTheoryIsBetter1stWP,include=FALSE-----------------------------------
#here I basically fit the bidding behavior with the theory and show that theory on risk is suggesting a risk aversion paramter close to .45y and that spite suggestes a spite-factor of 1.
#theory of spite explains data worse then theory on risk! (because alpha would need to be 1 to explain data)
##
spitelessFSP<-function(x){x/2}
spiteFSP <- function(x,alpha){ ((alpha+1)/(alpha+2))*x}
RiskFSP <- function(x,r){ x*(1/(1+r))}
##
data_for_testFSP<-daten[daten$FirstPrice==1,]
data_for_testFSP[,c("Spite_SVO","Spite_Quest","Spite_Kim","Risk_overall","rivalry","svo_type","IA","IAS")] <-
  scale(daten[daten$FirstPrice==1,c("Spite_SVO","Spite_Quest","Spite_Kim","Risk_overall","rivalry","svo_type","IA","IAS")])
##


## ----XXX----------------------------------------------------------------------
bidfunComplex<-function(v,a,r) {
    if(r==Inf)
        return(bidfun_Spite(v,a))
    if(a==0)
        return(bidfun_Risk(v,r))
    integrand <- function(x) {pmin(1e99,((-exp(a*x/r)+exp(a/r))^(1/(a-1)))/(exp(x/r)))}
    v<-pmin(.999,v)
  Part1<-integrate(integrand, lower = 0, upper = v, stop.on.error=FALSE)$value ## values can be huge
  Prep1<-((a-1))
  LogTop<- r*(a-1)*(exp(a/r)-1)^(-a/(a-1))
  LogButton<-((a^2)*Part1*(exp(a/r)-1)^(-a/(a-1))-(r*(a-1)))
  Add<-a*log(-exp((a*v/r))+exp((a/r)))
  Divide<-((a-1)*a)
  Full<- r*(Prep1*log(-LogTop/LogButton)+Add)/Divide
  return(pmin(1.5,Full)) ## limit bids in the same way as above
}
#
bidfun_Spite<-function(v,a){
    ##if(v==1){v<-.9999999999}#it can lead to issues for alpha>1 and v=1
    v<-pmin(.9999999999,v) 
    result<- (a+1)/(a*(2*a-1))*((1-a)*((1-v)^(a/(1-a))-1)+v*a)
    ##result2<-ifelse(result>1.5,1.5,result)
    ##return(result)
    return(pmax(0,pmin(1.5,result,1.5)))
}
##  
bidfun_Risk<-function(v,a) { ##if(v==1){v<-.9999999999}
    v<-pmin(.9999999999,v)
    integrand <- function(x) {a*(1-exp(-x/a))/(1-x)}
    result <- integrate(integrand, lower = 0, upper = v)$value
    pmin(1.5,result) ##; return(ifelse(result>1.5,1.5,result))}
}
##  


## ----WhichTheoryIsBetterAggregatesXXX-----------------------------------------
## measures for distance
## we did play with different distance measures, but no big effect, so let us concentrate on dL
dL <- function(x,y) 
    -logLik(lm(y ~ offset(x) -1))
##
dParamAR<-within(subset(daten,FirstPrice==0),{
    v<-variable/100
    b<-value/100
})
##                                               interval=c(.01,.99),m=m)$minimum)


## ----lmerCompare--------------------------------------------------------------
dParamARM <- dParamAR[,c("sid","gid","v","b")]
dParamARMV <- unique(dParamARM[["v"]])
lmerCompare <- function(ar) {
    bvEq<-dplyr::left_join(dParamARM,data.frame(v=dParamARMV,bEq=sapply(dParamARMV,bidfunComplex,a=ar[1],r=ar[2])),by="v")
    -logLik(suppressMessages(lmer(b ~ offset(bEq) - 1 + (1|sid) + (1|gid),data=bvEq)))
}
optAR<-optim(c(.3,.5),lmerCompare,hessian=TRUE)
optARRinf <- optimise(function(a) lmerCompare(c(a,Inf)),c(0.01,.99)) #why is it saying alpha= 0.1379 (loglik 18965.4) if you get a better loglik with alpha=.99999 (17880.73)?
optARa0   <- optimise(function(r) lmerCompare(c(0,r)),c(.01,100)) #same here: why is it saying that the best r is 0.4536201 (loglik 16390.52 ) if you get a better loglik with r=0.019 (loglik: 13151.62) ?
#Also: as both results basically tend to go into the extremes it can be interpreted as a sign that using model fit does not make much sense
pRinf <- pchisq(2 * (optARRinf$objective - optAR$value),df=1,lower.tail=FALSE)
pa0   <- pchisq(2 * (optARa0$objective   - optAR$value),df=1,lower.tail=FALSE)
optAR$CI <- optAR$par + sqrt(diag(solve(optAR$hessian))) %*% rbind(qnorm(c(.025,.975)))
optAR[["pval"]]<-c(pa0,pRinf)
for(i in 1:2)
    optAR[["citext"]][i]<-sprintf("%.3g~(CI_{95}=[%.3g,%.3g])",optAR$par[i],optAR$CI[i,1],optAR$CI[i,2])

#maybe a much simpler (and potentially appropriate approach) is to directly test what we are predicting:
#"Observation 1: For the case of uniformly distributed valuations in the second-price all-pay auction bids increase in spite for low valuations and they decrease in spite for high valuations" --> thus we could just see whether for low valuations the slope is increasing and for high valuations it is decreasing---if yes-- then theory of spite...if no then theory of risk


## ----aggregateBids1stWPand2ndAPOverbidding,fig.width=4------------------------
x<-seq(0,100,2.499)
lty<-c(1,3,2,4,6,5)
lwd<-c(1,1,1,1,3,1)
col<-rev(c("black","black","green","green","blue","blue"))
alphas<-list(0.0000001,.2,.9)
rs<-list(1,.1)
theory2<-ldply(alphas,function(a) as.data.frame(list(variable=x,overbid=100*sapply(x/100,function(x) spite(x,a)-spite(x,.00000001)),text=sprintf("Eq.~spite=$%g$",round(a,2)))))
theory2<-within(theory2,text<-revalue(text,c("Eq.~spite=$0$"="RNBE")))
theory3<-ldply(rs,function(a) as.data.frame(list(variable=x,overbid=100*sapply(x/100,function(x) Risk(x,a)-spiteless(x)),text=sprintf("Eq.~risk(CARA)=$%g$",round(a,2)))))
allPlot2<-rbind.fill(theory2,theory3,
           cbind(aggregate(overbid~variable,data=daten[daten$FirstPrice==0,],FUN=median),text="Observed Behavior"))
##
parS<-list(layout.widths=list(left.padding=0,right.padding=0,ylab.axis.padding=0,axis.key.padding=1))
parS<-c(parS,list(layout.heights=list(top.padding=0,axis.top=0,xlab=0)))
parS<-c(parS,list(parS,superpose.line = list(col=col,lty=lty,lwd=lwd)))
aKey <- list(corner=c(0,1),cex=.7,lines=TRUE,points=FALSE,size=2)
#
plot(xyplot(main="Second-price all-pay",overbid~variable,group=text,data=allPlot2,type="l",xlab="$v$",ylab="$b-b^\\IIAP$",ylim=c(-50,100),col=col, par.settings = parS, auto.key=aKey))
##


## ----aggregateBidsRiskAndSpiteOverbidding,fig.width=6-------------------------
x<-seq(0,100,2.499)
col=rev(c("darkgreen","red","darkblue","blue","black"))
lwd=c(1,1,1,3,3)
##
theory2<-ldply(list(0.0000001,.3,.9),function(a) as.data.frame(list(variable=x,overbid=100*sapply(x/100,function(x) spite(x,a)-spite(x,.00000001)),text=sprintf("Eq.~spite=%g",round(a,2)))))
allPlot2<-rbind.fill(theory2,
           cbind(aggregate(overbid~variable,data=subset(daten[daten$FirstPrice==0,],SpiteSum>median(SpiteSum)),FUN=median),text="Spite>Median"),
cbind(aggregate(overbid~variable,data=subset(daten[daten$FirstPrice==0,],SpiteSum<=median(SpiteSum)),FUN=median),text="Spite<Median"))
##
parS<-list(layout.widths=list(left.padding=0,right.padding=0,ylab.axis.padding=0,axis.key.padding=1))
parS<-c(parS,list(layout.heights=list(top.padding=0,axis.top=0,xlab=0)))
parS<-c(parS,list(parS,superpose.line = list(col=col,lty=lty,lwd=lwd)))
aKey <- list(corner=c(0,1),cex=.7,lines=TRUE,points=FALSE,size=2)
##
plot(xyplot(main="Spite",overbid~variable,group=text,data=allPlot2,type="l",xlab="$v$",ylab="$b-b^\\IIAP$",ylim=c(-50,100),col=col, par.settings = parS,auto.key=aKey),split=c(1,1,2,1),more=TRUE)
##
theory2<-ldply(list(3000,1,.1),function(a) as.data.frame(list(variable=x,overbid=100*sapply(x/100,function(x) Risk(x,a)-spiteless(x)),text=sprintf("Eq.~risk=%g",round(a,2)))))
allPlot2<-rbind.fill(theory2,
           cbind(aggregate(overbid~variable,data=subset(daten[daten$FirstPrice==0,],Risk_overall>median(Risk_overall)),FUN=median),text="Risk>Median"),
cbind(aggregate(overbid~variable,data=subset(daten[daten$FirstPrice==0,],Risk_overall<=median(Risk_overall)),FUN=median),text="Risk<Median"))
plot(xyplot(main="Risk",overbid~variable,group=text,data=allPlot2,type="l",xlab="$v$",ylab="$b-b^\\IIAP$",ylim=c(-50,100),col=col, par.settings = parS,auto.key=aKey),split=c(2,1,2,1),more=FALSE)


## ----AverageOverbiddingSimple,include=FALSE-----------------------------------
#########################################################################
#
# Here is loook at the average behavior over all rounds over subjects
#
#########################################################################
tmp2ndAp<-summarySE(data_for_test, measurevar="overbid", groupvars=c("Risk_overall","SpiteSum","gid","sid"))
tmp1stWP<-summarySE(data_for_testFSP, measurevar="overbid", groupvars=c("Risk_overall","SpiteSum","gid","sid"))
tmp2ndAp$FirstPrice<-0
tmp1stWP$FirstPrice<-1
##
SimpleDataAvg<-rbind(tmp2ndAp,tmp1stWP)
##
SimpleModelAvg<-lmer(overbid~SpiteSum*as.factor(FirstPrice) +Risk_overall*as.factor(FirstPrice) +(1|gid),SimpleDataAvg)
SimpleModelAvg2ndAP<-lmer(overbid~SpiteSum +Risk_overall +(1|gid),SimpleDataAvg[SimpleDataAvg$FirstPrice==0,])
#so what do we see if we would look at the very simple averages:
#on avergae spite is increasing ovverbidding in the 2ndAp and risk is doing the opposite
#in the 1stWP the opposite is happening:
SimpleModelAvgFSP<-lmer(overbid~SpiteSum +Risk_overall +(1|gid),SimpleDataAvg[SimpleDataAvg$FirstPrice==1,])


## ----RegressionTableOverbiddingSimple, cache=TRUE, cache.comments=FALSE, results='asis'----
stargazer(SimpleModelAvg2ndAP,#ci = TRUE,
          single.row = TRUE,#Model_Over4,#title="Regression for H1",
          dep.var.labels.include = FALSE,             style=  "aer",
          notes="\\pStarNotes",notes.append=FALSE,
          star.cutoffs=c(.1,.05,.01,.001),star.char = c("+","*", "**", "***"),
          digits       = 2,
          digits.extra = 2,float=FALSE,          
          column.labels   = c("\\Iipap{}","\\Fpa{}"),
          covariate.labels = c("Spite","Risk"),
          model.numbers          = FALSE#,column.labels = c("", "Mixed Effects Model","",""),
) 


## ----ModelComparisionCutoffs, cache=TRUE, cache.comments=FALSE, results='asis'----


myBs <- function(x,knots=90,...) {
  if(identical(knots,90)) {
    cbind(pmin(0,x-90)/90,pmax(0,x-90)/90)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_90<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(90),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)



myBs <- function(x,knots=80,...) {
  if(identical(knots,80)) {
    cbind(pmin(0,x-80)/80,pmax(0,x-80)/80)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_80<-gamm(overbid ~ 
                           + SpiteSum+s(variable,bs="tp") #default
                         +SpiteSum:myBs(variable,knots=c(80),degree=1)
                         +Period,random=list(sid=~1,gid=~1),
                         data=data_for_test)









myBs <- function(x,knots=70,...) {
  if(identical(knots,70)) {
    cbind(pmin(0,x-70)/70,pmax(0,x-70)/70)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_70<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(70),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)











myBs <- function(x,knots=60,...) {
  if(identical(knots,60)) {
    cbind(pmin(0,x-60)/60,pmax(0,x-60)/60)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_60<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(60),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)













myBs <- function(x,knots=50,...) {
  if(identical(knots,50)) {
    cbind(pmin(0,x-50)/50,pmax(0,x-50)/50)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_50<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(50),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)







myBs <- function(x,knots=40,...) {
  if(identical(knots,40)) {
    cbind(pmin(0,x-40)/40,pmax(0,x-40)/40)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_40<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(40),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)






myBs <- function(x,knots=30,...) {
  if(identical(knots,30)) {
    cbind(pmin(0,x-30)/30,pmax(0,x-30)/30)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_30<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(30),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)






myBs <- function(x,knots=20,...) {
  if(identical(knots,20)) {
    cbind(pmin(0,x-20)/20,pmax(0,x-20)/20)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_20<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(20),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)






myBs <- function(x,knots=10,...) {
  if(identical(knots,10)) {
    cbind(pmin(0,x-10)/10,pmax(0,x-10)/10)
  }
  else {
    bs(x,knots=knots,...)
  }}


Model_Over2_Gam_BS_10<-gamm(overbid ~ 
                              + SpiteSum+s(variable,bs="tp") #default
                            +SpiteSum:myBs(variable,knots=c(10),degree=1)
                            +Period,random=list(sid=~1,gid=~1),
                            data=data_for_test)






# summary(Model_Over2_Gam_BS_90$lme)["logLik"]
# summary(Model_Over2_Gam_BS_80$lme)["logLik"]
# summary(Model_Over2_Gam_BS_70$lme)["logLik"]
# summary(Model_Over2_Gam_BS_60$lme)["logLik"]
# summary(Model_Over2_Gam_BS_50$lme)["logLik"]
# summary(Model_Over2_Gam_BS_40$lme)["logLik"]
# summary(Model_Over2_Gam_BS_30$lme)["logLik"]
# summary(Model_Over2_Gam_BS_20$lme)["logLik"]
# summary(Model_Over2_Gam_BS_10$lme)["logLik"]
# 
# 
# 
# 
# 
# 
# summary(Model_Over2_Gam_BS_90$lme)["AIC"]
# summary(Model_Over2_Gam_BS_80$lme)["AIC"]
# summary(Model_Over2_Gam_BS_70$lme)["AIC"]
# summary(Model_Over2_Gam_BS_60$lme)["AIC"]
# summary(Model_Over2_Gam_BS_50$lme)["AIC"]
# summary(Model_Over2_Gam_BS_40$lme)["AIC"]
# summary(Model_Over2_Gam_BS_30$lme)["AIC"]
# summary(Model_Over2_Gam_BS_20$lme)["AIC"]
# summary(Model_Over2_Gam_BS_10$lme)["AIC"]
# #so the best is about 50!
Cutoff50vs60<-lrtest(Model_Over2_Gam_BS_50$lme,Model_Over2_Gam_BS_60$lme)
Cutoff50vs70<-lrtest(Model_Over2_Gam_BS_50$lme,Model_Over2_Gam_BS_70$lme)
Cutoff50vs80<-lrtest(Model_Over2_Gam_BS_50$lme,Model_Over2_Gam_BS_80$lme)
Cutoff50vs90<-lrtest(Model_Over2_Gam_BS_50$lme,Model_Over2_Gam_BS_90$lme)



## ----myBS,include=FALSE-------------------------------------------------------
myBs <- function(x,knots=50,...) {
    if(identical(knots,50)) {
        cbind(pmin(0,x-50)/50,pmax(0,x-50)/50)
    }
    else {
        bs(x,knots=knots,...)
    }}

## ----RegressionGAM_BS,include=FALSE-------------------------------------------
Model_Over5_Gam_BS<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_test)

Model_Over4_Gam_BS<-gamm(overbid ~ Risk_overall+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_test)

Model_Over3_Gam_BS<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_test)

Model_Over2_Gam_BS<-gamm(overbid ~ 
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_test)
#
Model_Over1_Gam_BS<-gamm(overbid ~ s(variable,bs="tp") #default
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_test)


#################################################################
#
# Below I checked whether our results prevail if we use different cutoffs
# In summary: for cutoffs 40-80 everything remains basically as in the paper -- for 90 the results are still very similar but lose significance
#
#################################################################

   #Combined Measure: for 40 it is sign!
    #Combined Measure: for 50 it is sign!
    #Combined Measure: for 60 it is sign!
    #Combined Measure: for 70 it is sign!
    #Combined Measure: for 80 it is marg sign! (the first is marg sig and the second is  very large negative but not sig!)
    #Combined Measure: for 90 the estimates are similar (and the sign prevails...but the results are not sig anymore)
    
    
    
    #Own Measure: for 40 it is sign! [the first, not the second]
    #Own Measure: for 50 it is sign! [the first, not the second]
    #Own Measure: for 60 it is sign! [the first, not the second]
    #Own Measure: for 70 it is sign! [the first, not the second]
    #Own Measure: for 80 it is marg sign! (spite bevore is marg sig and after it is  very large negative but not sig!)
    #Own Measure: for 90 it is marg sign! (spite bevore is marg sig and after it is  very large negative but not sig!)
    
 
    
    
    #Eric Measure: for 40 it is sign!  [the first, not the second]
    #Eric Measure: for 50 it is sign!  [the first, not the second]
    #Eric Measure: for 60 it is sign!  [the first, not the second]
    #Eric Measure: for 70 it is sign!  [the first, not the second]
    #Eric Measure: for 80 it is marg sign! (spite bevore is marg sig and after it is  very large negative but not sig!)
    #Eric Measure: for 90 it is NOT sign!
    
       

    
    
    #Quest Measure: for 40 it is marg sign!  [the second, not the first]
    #Quest Measure: for 50 it is NOT sign! 
    #Quest Measure: for 60 it is NOT sign!  
    #Quest Measure: for 70 it is NOT sign!
    #Quest Measure: for 80 it is NOT sign!
    #Quest Measure: for 90 it is NOT sign! 


## ----RegressionGam_BS,error=FALSE,warning=FALSE,results='asis'----------------

mycoeffnames<-c("Period","SpiteSum",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)1",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)2",
                "Risk_overall",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)1",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)2",
                            "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
                            "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
                            "Bayesian Inf. Crit.")
ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
newnames=c( "Period","Spite","\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
            "Risk","\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}","Male", "Rivalry","SVO","IA","Constant")


PrintstargazerGam(list(Model1=Model_Over1_Gam_BS,Model2=Model_Over2_Gam_BS,Model3=Model_Over3_Gam_BS,
                       Model4=Model_Over4_Gam_BS,Model5=Model_Over5_Gam_BS),
             multipleObjects=TRUE,
             mycoeffnames=mycoeffnames,
             ModelNames=ModelNames,newnames=newnames
             )




## ----iipapCount,include=FALSE-------------------------------------------------
iipapCount<-table(aggregate(Period ~ sid,data=data_for_test,FUN=max)[["Period"]])
if(length(iipapCount)>2) 
    stop("length of experiment more diverse than expected")


## ----fig.Spline.Overbidding---------------------------------------------------
newd <- function (s) within(data_for_test[1:11,],{SpiteSum=s;Measure="Spite"})
newdRisk <- function (s) within(data_for_test[1:11,],{Risk_overall=s;Measure="Risk"})
newdata<-rbind(rbind.fill(lapply(c(-5,0,5),newd)),rbind.fill(lapply(c(-5,0,5),newdRisk)))
newdata<-within(newdata,{variable[variable==20]<-25;variable[variable==70]<-75});
overSpline <- function(model,text,Measure) {
  within(newdata[newdata$Measure==Measure,],{overbid<-predict(model,newdata=newdata[newdata$Measure==Measure,])
  text<-text})}
allOverData<-rbind.fill(overSpline(Model_Over1_Gam_BS$gam,"$C'_1$","Spite"),overSpline(Model_Over2_Gam_BS$gam,"$C'_2$","Spite"),overSpline(Model_Over3_Gam_BS$gam,"$C'_3$","Spite"),
                        overSpline(Model_Over4_Gam_BS$gam,"$C'_4$","Risk"),overSpline(Model_Over5_Gam_BS$gam,"$C'_5$","Risk"))
allOverData1<-allOverData[allOverData$Measure=="Spite",]
allOverData2<-allOverData[allOverData$Measure!="Spite",]
allOverData1<-rename(allOverData1,c("SpiteSum"="Spite"))
allOverData2<-rename(allOverData2,c("Risk_overall"="Spite"))
allOverData<-rbind(allOverData1[,c("overbid","variable","Spite","text")],allOverData2[,c("overbid","variable","Spite","text")])
xyplot(overbid~variable | text,group=Spite,data=allOverData,type="l",ylab="$s(v)$",xlab="$v$",as.table=TRUE,auto.key=list(title="Spite ($C'_2$,$C'_3$) \n Risk ($C'_4$,$C'_5$):",corner=c(1.05,.00),lines=TRUE,points=FALSE,cex.title=.8,cex=.8))


## ----overTime-----------------------------------------------------------------
overTimeFun <- function(data,formula,coefname,type,fun=function(...) lmer(...)) 
    cbind(ddply(data,.(Period),function(d) c(beta=unname(fixef(fun(formula,data=d))[coefname]))),type=type)
#
overTimeSPA<-list()
#
overTimeSPA[["SpiteV"]]<-overTimeFun(
    data=data_for_test,
    formula=overbid ~ SpiteSum+s(variable,bs="tp")+SpiteSum:myBs(variable,knots=c(50),degree=1),
    coefname=paste("XSpiteSum:myBs(variable, knots = c(50), degree = 1)",1:2,sep=""),
    type="$\\IIAP$, $C'_2$, Spite$\\times v$",
    fun=function(...) gamm(...,random=list(sid=~1,gid=~1))[["lme"]])
#
overTimeSPA[["RiskV"]]<-overTimeFun(
    data=data_for_test,
    formula=overbid ~ Risk_overall+s(variable,bs="tp") +Risk_overall:myBs(variable,knots=c(50),degree=1),
    coefname=paste("XRisk_overall:myBs(variable, knots = c(50), degree = 1)",1:2,sep=""),
    type="$\\IIAP$, $C'_4$, Risk$\\times v$",
    fun=function(...) gamm(...,random=list(sid=~1,gid=~1))[["lme"]])

## ----overTimePlot,fig.width=6.5,external=FALSE--------------------------------
key=list(lines=TRUE,columns=2,size=2,text=c("[0,50]","[50,100]"),cex=.7,between=1)
overTimeSPA.plot<-xyplot(beta1+beta2~Period | type,main="\\Iipap, Eq. \\eqref{eq:overMix}",data=subset(rbind.fill(overTimeSPA),Period<=15),ylab="",auto.key=key,type=c("p","r"),par.strip.text=list(cex=.7))+layer(panel.refline(h=0))
plot(overTimeSPA.plot)


## ----Individual rationality,error=FALSE,include=FALSE,warning=FALSE-----------
data_for_test<-daten[daten$FirstPrice==0,]


## ----holtLauryLotteries,results='asis'----------------------------------------
xx<-data.frame(t(sapply(1:10,function(i) c(sprintf("In %d out of 10 cases you will earn 1800 points and in %d out of 10 cases you will earn 1440 points",i,10-i),
sprintf("In %d out of 10 cases you will earn 3465 points and in %d out of 10 cases you will earn 90 points",i,10-i)))))
names(xx)<-c("Lottery A","Lottery B")
print(xtable(xx,align=c("c","p{.45\\linewidth}","p{.45\\linewidth}")),include.rownames=FALSE,hline.after=seq(-1,10))


## ----TableOwnSpite, cache=TRUE, cache.comments=FALSE, results='asis', out.width="0.1\\linewidth"----
SpiteTable<- data.frame(Colnames= character(7), NonSpite= integer(7), Spite = integer(7),MeanSpite= numeric(7))
sbj2ndAP<-sbj
NsubjFull<-Nsubj+Nsubj2
SpiteTable$Colnames<-c("IA","IA-WP","RG","RG-WP","PS","PS-WP","$\\sum$")
SpiteTable$NonSpite<-c(sum(sbj2ndAP$Spitefullness_inequality==0),sum(sbj2ndAP$Spitefullness_inequality_willingtoPay==0),
                       sum(sbj2ndAP$Spitefullness_relativeGain==0),sum(sbj2ndAP$Spitefullness_relativeGain_willingtoPay==0),
                       sum(sbj2ndAP$Spitefullness_pure==0),sum(sbj2ndAP$Spitefullness_pure_willingtoPay==0),
                       sum(sbj2ndAP$Spitefullness_overall==0))
SpiteTable$NonSpite<-round(SpiteTable$NonSpite/NsubjFull*100,0)
SpiteTable$Spite<-100-SpiteTable$NonSpite
SpiteTable$MeanSpite<-c(mean(sbj2ndAP$Spitefullness_inequality),mean(sbj2ndAP$Spitefullness_inequality_willingtoPay),
                       mean(sbj2ndAP$Spitefullness_relativeGain),mean(sbj2ndAP$Spitefullness_relativeGain_willingtoPay),
                       mean(sbj2ndAP$Spitefullness_pure),mean(sbj2ndAP$Spitefullness_pure_willingtoPay),
                       mean(sbj2ndAP$Spitefullness_overall))
colnames(SpiteTable) <- c("Submeasures", "No Spite in \\%", "Spite in \\%", "Average Spite")
Spite.table<-xtable(SpiteTable)
align(Spite.table) <- "|l|l|rrr|"
print(xtable(Spite.table),include.rownames=FALSE,booktabs=TRUE,hline.after=c(-1,0,6,7))
#note that both times we had very very similar results in spite and risk:


## ----zTree.data.KR,results="asis"---------------------------------------------
ddKR<-data.frame(list(x=seq(500,1000,50),
                    y=c(0,10,20,40,80,160,320,500,700,900,1000)))
fbExKR<-data.frame(list(auctions=c(9,10,6,5,7,2,4,8,1,3),
                    valuationsReal=c(511,532,538,570,607,653,747,836,867,913),
                    bidsReal=c(2,6,8,14,23,42,155,414,568,752),
                    valuationsOther=rep("?",10),
                    bidsOther=c(715,NA,942,916,48,NA,NA,NA,NA,NA)))
fbExKR<-within(fbExKR,Result<-ifelse(is.na(bidsOther),"Won","Lost"))


## ----zTree.input.KR,results='asis'--------------------------------------------
with(ddKR,print(xtable(rbind(`Your valuation`=x,
                           `Your bid`=sprintf("\\INPUTOLIVER{%d}",y)),align=rep('c',length(x)+1)),
      include.colnames=FALSE,hline.after=NULL))


## ----zTree.plot.KR------------------------------------------------------------
par(mar=c(4,4,.5,.7))
plot(y~x,ylim=c(0,1500),data=ddKR,t="l",las=1,xaxs='i',xaxp=c(500,1000,10),yaxs='i',yaxp=c(0,1500,15),xlab="Valuation",ylab="Bid",col="blue",lwd=3)
abline(v=seq(500,1000,50));abline(h=seq(0,1500,100))


## ----prepare.zTree.plot.feedback,results='asis'-------------------------------
with(fbExKR,
     print(xtable(rbind(`Auction`=auctions,
           `Your valuation`=valuationsReal,
           `Your bid`=bidsReal,
           `Other's valuation`=as.character(valuationsOther),
           `Other's bid`=ifelse(is.na(bidsOther),'smaller',bidsOther),
           `Won/lost`=ifelse(Result=="Lost","\\color{red} lost","\\color{green} won")),
           align=c('|l|',rep('c|',length(auctions)))),
      include.colnames=FALSE,hline.after=0:6))


## ----zTree.plot.feedback.KR,fig.width=6---------------------------------------
par(mar=c(4,4,.3,5.5))
plot(y~x,ylim=c(0,1500),xlim=c(500,1000),data=ddKR,t="l",las=1,xaxs='i',xaxp=c(500,1000,10),yaxs='i',yaxp=c(0,1500,15),xlab="Your valuation",ylab="Bid",bty='n',lwd=3)
abline(v=seq(500,1000,50));
abline(h=seq(0,1500,100));
par(xpd=NA)
with(fbExKR,{
points(valuationsReal,bidsOther,col="blue",pch=4,lwd=2)
points(valuationsReal,bidsReal,col=ifelse(Result=='Lost',"red",'green'),pch=5,lwd=2)
text(1000,max(ddKR[["y"]]),"Your bid",pos=4)
text(1000,max(ddKR[["y"]])/2,"\\begin{tabular}{@{}l@{}}Other's bid\\\\when he wins\\end{tabular}",pos=4,col='blue')
text(valuationsReal,bidsReal,auctions,adj=c(-.5,1),cex=.5)
})
legend(x=400,y=-150,bg="white",c("Bids in auctions lost","Bids in auctions won"),pch=5,col=c("red","green"),bty="n")


## ----zTree.bidAdaptation------------------------------------------------------
bidAdaptEx<-data.frame(list(
    pos=1:10,
    round=c(10,2,4,8,1,3,9,6,5,7),
    valuationsReal=c(532,653,747,836,867,913,511,538,570,607),
    bidsReal=c(6,42,155,444,568,752,2,8,14,23),
    bidsOther=c(rep(NA,6),c(715,942,916,48))))
bidAdaptEx<-within(bidAdaptEx,
    bidsNew<-ceiling(c(bidsReal[1:6]*1.26,c(273,363,357,32))))

## ----zTree.plot.bidAdaptation,fig.height=3,fig.width=6.5----------------------
par(mar=c(2,3,1.5,6.5))
plot(NULL,xlim=c(1,10),ylim=c(0,1500),xaxs='i',yaxs='i',axes=FALSE,xlab="",ylab="Bid")
abline(v=1:10)
abline(h=seq(0,1500,100))
abline(v=6.5,lwd=3)
#       
with(bidAdaptEx,{axis(1,at=pos,labels=round);
    axis(2,at=seq(0,1500,100),las=1)})
with(subset(bidAdaptEx,is.na(bidsOther)),{
    points(pos,bidsReal,col="green")
    lines(pos,bidsReal,col="green")
    lines(pos,bidsNew,col="blue")
    } )
q<-apply(subset(bidAdaptEx,is.na(bidsOther)),1,FUN=function(x) lines(c(x["pos"],x["pos"]),c(x["bidsReal"],x["bidsNew"]),col="green",lwd=2))
with(subset(bidAdaptEx,!is.na(bidsOther)),{
    lines(pos,bidsReal,col="red")
    lines(pos,bidsNew,col="magenta")
    points(pos,bidsOther,col="blue",pch=4)
  })
q<-apply(subset(bidAdaptEx,!is.na(bidsOther)),1,FUN=function(x) lines(c(x["pos"],x["pos"]),c(x["bidsNew"],x["bidsOther"]),col="red",lwd=2))
par(xpd=NA)
text(10,-50,"Round",pos=4)
text(10,200,"\\begin{tabular}{@{}l}Your bids so far\\\\(lost)\\end{tabular}",pos=4,col='red')
text(10,400,"\\begin{tabular}{@{}l}Your bids so far\\\\(won)\\end{tabular}",pos=4,col='green')
text(10,650,"\\begin{tabular}{@{}l}Bids of the other\\\\if he wins\\end{tabular}",pos=4,col='blue')
text(10,1350,"Your new bids",pos=4,col='magenta')
text(3.5,1500,"Won",pos=3,col='green')
text(8.5,1500,"Lost",pos=3,col='red')


## ----zTree.table.bidAdaptation,results='asis'---------------------------------
with(bidAdaptEx,print(xtable(rbind(Round=round,
                                   `Your valuation`=valuationsReal,
                                   `Bid (so far)`=bidsReal,
                                   `Bid (new)`=bidsNew),
                             align=rep('c',length(round)+1),digits=0),include.colnames=FALSE,hline.after=NULL))


## ----Regression1Data,include=FALSE--------------------------------------------
set.seed(123)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model5
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model5<-lmer(value ~ SpiteSum +
                            Geschlecht + Risk_overall*myBs(variable,knots=c(50),degree=1) +rivalry +SVO+I(IA+IAS)+
                                (1|sid) +Period+(1|gid)
                              ,data=data_for_test)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model4
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model4<-lmer(value ~ Risk_overall*myBs(variable,knots=c(50),degree=1)+
                                (1|sid) +Period+(1|gid)
                              ,data=data_for_test)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model3 
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model3<-lmer(value ~ SpiteSum*myBs(variable,knots=c(50),degree=1) +
                            Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                                (1|sid) +Period+(1|gid)
                              ,data=data_for_test)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model2
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model2<-lmer(value ~ SpiteSum*myBs(variable,knots=c(50),degree=1)+
                                (1|sid) +Period+(1|gid) 
                              ,data=data_for_test)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model1 
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model1<-lmer(value ~ (1|sid) +Period+(1|gid)+myBs(variable,knots=c(50),degree=1) 
                              ,data=data_for_test)


## ----Regression1, cache=TRUE, cache.comments=FALSE, results='asis', out.width="0.1\\linewidth"----
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Using Stargazer to make a regression table
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
stargazer(Model1,Model2,Model3,Model4,Model5,#ci = TRUE,
          notes="\\pStarNotes",notes.append=FALSE,
          star.cutoffs=c(.1,.05,.01,.001),star.char = c("+","*", "**", "***"),
          single.row = TRUE,#title="Regression for H1",
          dep.var.labels.include = FALSE,   style=  "aer",    
          digits       = 2,
          digits.extra = 2,float=FALSE,
          column.labels   = c("$C_1$","$C_2$","$C_3$","$C_4$","$C_5$"),
          order = c(2,7,8,1,3,4,10,5,6),
          covariate.labels = c( "Period","$\\vLow$","$\\vHigh$",
                                "Spite",
                                "\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
                                "Risk",
                                "\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}",
                                "Male","Rivalry","SVO","IA"),column.sep.width="-8pt",
          model.numbers          = FALSE#,column.labels = c("", "Mixed Effects Model","",""),
)    


## ----Regression2Data,include=FALSE,eval=F-------------------------------------
## set.seed(123)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model5
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over5<-lmer(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                     + SpiteSum+ Risk_overall:myBs(variable,knots=c(50),degree=1)+myBs(variable,knots=c(25,50,75),degree=1)
##                   +(1|sid) +Period+(1|gid),
##                   data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model4
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over4<-lmer(overbid ~ Risk_overall+Risk_overall:myBs(variable,knots=c(50),degree=1)+myBs(variable,knots=c(25,50,75),degree=1)
##                   +(1|sid) +Period+(1|gid),
##                   data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model3
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over3<-lmer(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                     + SpiteSum+ SpiteSum:myBs(variable,knots=c(50),degree=1)+myBs(variable,knots=c(25,50,75),degree=1)
##                   +(1|sid) +Period+(1|gid),
##                   data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model2
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over2<-lmer(overbid ~ #Geschlecht + Risk_overall +rivalry +SVO+#I(IA+IAS)+
##                     SpiteSum+ SpiteSum:myBs(variable,knots=c(50),degree=1)+myBs(variable,knots=c(25,50,75),degree=1)
##                   +(1|sid) +Period+(1|gid),
##                   data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model1
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over1<-lmer(overbid ~ #Geschlecht + Risk_overall +rivalry +svo_type +#SVO+I(IA+IAS)+
##                     + SpiteSum + myBs(variable,knots=c(25,50,75),degree=1)
##                   +(1|sid) +Period+(1|gid),
##                   data=data_for_test)


## ----Regression2LinearSpline, cache=TRUE, cache.comments=FALSE, results='asis', out.width="0.1\\linewidth",eval=F----
## stargazer(Model_Over1,Model_Over2,Model_Over3,Model_Over4,Model_Over5,#ci = TRUE,
##           single.row = TRUE,#Model_Over4,#title="Regression for H1",
##           dep.var.labels.include = FALSE,             style=  "aer",
##           notes="\\pStarNotes",notes.append=FALSE,
##           star.cutoffs=c(.1,.05,.01,.001),star.char = c("+","*", "**", "***"),
##           digits       = 2,
##           digits.extra = 2,float=FALSE,
##           column.labels   = c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$"),
##           order=c(11,7:10,6,12,13,2,14,15),
##           covariate.labels = c( "Period","Valuation$_{25}$","Valuation$_{50}$",
##                                 "Valuation$_{75}$","Valuation$_{100}$",
##                                 "Spite",
##                                 "\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
##                                 "Risk",
##                                 "\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}",
##                                 "Male","Rivalry","SVO","IA"),
##           model.numbers          = FALSE#,column.labels = c("", "Mixed Effects Model","",""),
## )


## ----RegressionBSplines,include=FALSE,eval=F----------------------------------
## set.seed(123)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model5
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over5_BSpline<-lmer(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum
##                        +Risk_overall:myBs(variable,knots=c(50),degree=1)+
##                        bs(variable)+Period+(1|sid)+(1|gid), data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model4
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over4_BSpline<-lmer(overbid ~ Risk_overall
##                        +Risk_overall:myBs(variable,knots=c(50),degree=1)+
##                        bs(variable)+Period+(1|sid)+(1|gid), data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model3
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over3_BSpline<-lmer(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum
##                        +SpiteSum:myBs(variable,knots=c(50),degree=1)+
##                        bs(variable)+Period+(1|sid)+(1|gid), data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model2
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over2_BSpline<-lmer(overbid ~ SpiteSum
##                        +SpiteSum:myBs(variable,knots=c(50),degree=1)+
##                        bs(variable)+Period+(1|sid)+(1|gid), data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model1
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over1_BSpline<-lmer(overbid ~ bs(variable)+Period+(1|sid)+(1|gid), data=data_for_test)


## ----Regression2BSpline, cache=TRUE, cache.comments=FALSE, results='asis', out.width="0.1\\linewidth",eval=F----
## stargazer(Model_Over1_BSpline,Model_Over2_BSpline,Model_Over3_BSpline,Model_Over4_BSpline,Model_Over5_BSpline,#ci = TRUE,
##           single.row = TRUE,#Model_Over4,#title="Regression for H1",
##           dep.var.labels.include = FALSE,             style=  "aer",
##           notes="\\pStarNotes",notes.append=FALSE,
##           star.cutoffs=c(.1,.05,.01,.001),star.char = c("+","*", "**", "***"),
##           digits       = 2,
##           digits.extra = 2,float=FALSE,
##           column.labels   = c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$"),
##           order = c(10,7,8,9,6,11,12,2,13,14),
##           covariate.labels = c( "Period","Valuation$_{33}$","Valuation$_{66}$","Valuation$_{100}$",
##                                 "Spite",
##                                 "\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
##                                 "Risk",
##                                 "\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}",
##                                 "Male","Rivalry","SVO","IA"),
##           model.numbers          = FALSE#,column.labels = c("", "Mixed Effects Model","",""),
## )


## ----RegressionGAMCC,include=FALSE,eval=F-------------------------------------
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model5
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over5_GamCC<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum+s(variable,bs="cc") #default
##                        +Risk_overall:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model4
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over4_GamCC<-gamm(overbid ~ Risk_overall+s(variable,bs="cc") #default
##                        +Risk_overall:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model3
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over3_GamCC<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum+s(variable,bs="cc") #default
##                        +SpiteSum:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model2
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over2_GamCC<-gamm(overbid ~
##                          + SpiteSum+s(variable,bs="cc") #default
##                        +SpiteSum:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model1
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over1_GamCC<-gamm(overbid ~ s(variable,bs="cc") #default
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)


## ----RegressionGamCC,error=FALSE,warning=FALSE,results='asis',eval=F----------
## 
## mycoeffnames<-c("Period","SpiteSum",
##                 "SpiteSum:myBs(variable, knots = c(50), degree = 1)1",
##                 "SpiteSum:myBs(variable, knots = c(50), degree = 1)2",
##                 "Risk_overall",
##                 "Risk_overall:myBs(variable, knots = c(50), degree = 1)1",
##                 "Risk_overall:myBs(variable, knots = c(50), degree = 1)2",
##                             "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
##                             "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
##                             "Bayesian Inf. Crit.")
## ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
## newnames=c( "Period","Spite","\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
##             "Risk","\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}","Male", "Rivalry","SVO","IA","Constant")
## 
## PrintstargazerGam(list(Model1=Model_Over1_GamCC,Model2=Model_Over2_GamCC,Model3=Model_Over3_GamCC,
##                        Model4=Model_Over4_GamCC,Model5=Model_Over5_GamCC),
##              multipleObjects=TRUE,
##              mycoeffnames=mycoeffnames,
##              ModelNames=ModelNames,newnames=newnames)
## 
## 


## ----RegressionGAMPS,include=FALSE,eval=F-------------------------------------
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model5
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over5_GamPS<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum+s(variable,bs="ps") #default
##                        +Risk_overall:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model4
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over4_GamPS<-gamm(overbid ~ Risk_overall+s(variable,bs="ps") #default
##                        +Risk_overall:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model3
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over3_GamPS<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum+s(variable,bs="ps") #default
##                        +SpiteSum:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model2
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over2_GamPS<-gamm(overbid ~
##                          + SpiteSum+s(variable,bs="ps") #default
##                        +SpiteSum:myBs(variable,knots=c(50),degree=1)
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model1
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over1_GamPS<-gamm(overbid ~ s(variable,bs="ps") #default
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)


## ----RegressionGamPS,error=FALSE,warning=FALSE,results='asis',eval=F----------
## 
## mycoeffnames<-c("Period","SpiteSum",
##                 "SpiteSum:myBs(variable, knots = c(50), degree = 1)1",
##                 "SpiteSum:myBs(variable, knots = c(50), degree = 1)2",
##                 "Risk_overall",
##                 "Risk_overall:myBs(variable, knots = c(50), degree = 1)1",
##                 "Risk_overall:myBs(variable, knots = c(50), degree = 1)2",
##                             "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
##                             "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
##                             "Bayesian Inf. Crit.")
## ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
## newnames=c( "Period","Spite","\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
##             "Risk","\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}","Male", "Rivalry","SVO","IA","Constant")
## 
## PrintstargazerGam(list(Model1=Model_Over1_GamPS,Model2=Model_Over2_GamPS,Model3=Model_Over3_GamPS,
##                        Model4=Model_Over4_GamPS,Model5=Model_Over5_GamPS),
##              multipleObjects=TRUE,
##              mycoeffnames=mycoeffnames,
##              ModelNames=ModelNames,newnames=newnames)
## 
## 


## ----RegressionGAMPoly,include=FALSE,eval=F-----------------------------------
## data_for_test$variableSQ<-data_for_test$variable^2
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model5
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over5_Gam_Poly2<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum+s(variable,bs="tp") #default
##                        +Risk_overall:variable+Risk_overall:variableSQ
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model4
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over4_Gam_Poly2<-gamm(overbid ~ Risk_overall+s(variable,bs="tp") #default
##                        +Risk_overall:variable+Risk_overall:variableSQ
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model3
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over3_Gam_Poly2<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
##                          + SpiteSum+s(variable,bs="tp") #default
##                        +SpiteSum:variable+SpiteSum:variableSQ
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model2
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over2_Gam_Poly2<-gamm(overbid ~
##                          + SpiteSum+s(variable,bs="tp") #default
##                        +SpiteSum:variable+SpiteSum:variableSQ
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## # Model1
## #%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
## Model_Over1_Gam_Poly2<-gamm(overbid ~ s(variable,bs="tp") #default
##                        +Period,random=list(sid=~1,gid=~1),
##                        data=data_for_test)
## 


## ----RegressionGam_BS_Poly,error=FALSE,warning=FALSE,results='asis',eval=F----
## mycoeffnames<-c("Period","SpiteSum",
##                 "SpiteSum:variable",
##                 "SpiteSum:variableSQ",
##                 "Risk_overall",
##                 "Risk_overall:variable",
##                 "Risk_overall:variableSQ",
##                             "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
##                             "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
##                             "Bayesian Inf. Crit.")
## ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
## newnames=c( "Period","Spite","Spite $\\times v$","Spite $\\times$ $v^2$",
##             "Risk","Risk $\\times v$","Risk $\\times$ $v^2$","Male", "Rivalry","SVO","IA","Constant")
## 
## 
## PrintstargazerGam(list(Model1=Model_Over1_Gam_Poly2,Model2=Model_Over2_Gam_Poly2,Model3=Model_Over3_Gam_Poly2,
##                        Model4=Model_Over4_Gam_Poly2,Model5=Model_Over5_Gam_Poly2),
##              multipleObjects=TRUE,
##              mycoeffnames=mycoeffnames,
##              ModelNames=ModelNames,newnames=newnames
##              )
## 
## 


## ----RegressionGAM_BS_Kim,include=FALSE---------------------------------------
data_for_testTMP<-data_for_test
data_for_testTMP$SpiteSum<-data_for_testTMP$Spite_Kim
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model5
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over5_Gam_BS_Kim<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model4
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over4_Gam_BS_Kim<-gamm(overbid ~ Risk_overall+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model3
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over3_Gam_BS_Kim<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model2
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over2_Gam_BS_Kim<-gamm(overbid ~ 
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model1
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over1_Gam_BS_Kim<-gamm(overbid ~ s(variable,bs="tp") #default
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)



## ----RegressionGam_BS_Kim,error=FALSE,warning=FALSE,results='asis'------------

mycoeffnames<-c("Period","SpiteSum",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)1",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)2",
                "Risk_overall",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)1",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)2",
                            "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
                            "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
                            "Bayesian Inf. Crit.")
ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
newnames=c( "Period","Spite","\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
            "Risk","\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}","Male", "Rivalry","SVO","IA","Constant")


PrintstargazerGam(list(Model1=Model_Over1_Gam_BS_Kim,Model2=Model_Over2_Gam_BS_Kim,Model3=Model_Over3_Gam_BS_Kim,
                       Model4=Model_Over4_Gam_BS_Kim,Model5=Model_Over5_Gam_BS_Kim),
             multipleObjects=TRUE,
             mycoeffnames=mycoeffnames,
             ModelNames=ModelNames,newnames=newnames
             )




## ----RegressionGAM_BS_SVO,include=FALSE---------------------------------------
data_for_testTMP<-data_for_test
data_for_testTMP$SpiteSum<-data_for_testTMP$Spite_SVO
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model5
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over5_Gam_BS_SVO<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model4
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over4_Gam_BS_SVO<-gamm(overbid ~ Risk_overall+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model3
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over3_Gam_BS_SVO<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model2
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over2_Gam_BS_SVO<-gamm(overbid ~ 
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model1
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over1_Gam_BS_SVO<-gamm(overbid ~ s(variable,bs="tp") #default
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)



## ----RegressionGam_BS_SVO,error=FALSE,warning=FALSE,results='asis'------------

mycoeffnames<-c("Period","SpiteSum",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)1",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)2",
                "Risk_overall",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)1",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)2",
                            "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
                            "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
                            "Bayesian Inf. Crit.")
ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
newnames=c( "Period","Spite","\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
            "Risk","\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}","Male", "Rivalry","SVO","IA","Constant")


PrintstargazerGam(list(Model1=Model_Over1_Gam_BS_SVO,Model2=Model_Over2_Gam_BS_SVO,Model3=Model_Over3_Gam_BS_SVO,
                       Model4=Model_Over4_Gam_BS_SVO,Model5=Model_Over5_Gam_BS_SVO),
             multipleObjects=TRUE,
             mycoeffnames=mycoeffnames,
             ModelNames=ModelNames,newnames=newnames
             )




## ----RegressionGAM_BS_Quest,include=FALSE-------------------------------------
data_for_testTMP<-data_for_test
data_for_testTMP$SpiteSum<-data_for_testTMP$Spite_Quest
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model5
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over5_Gam_BS_Quest<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model4
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over4_Gam_BS_Quest<-gamm(overbid ~ Risk_overall+s(variable,bs="tp") #default
                       +Risk_overall:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model3
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over3_Gam_BS_Quest<-gamm(overbid ~ Geschlecht + Risk_overall +rivalry +SVO+I(IA+IAS)+
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model2
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over2_Gam_BS_Quest<-gamm(overbid ~ 
                         + SpiteSum+s(variable,bs="tp") #default
                       +SpiteSum:myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
# Model1
#%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Model_Over1_Gam_BS_Quest<-gamm(overbid ~ s(variable,bs="tp") #default
                       +Period,random=list(sid=~1,gid=~1),
                       data=data_for_testTMP)


## ----RegressionGam_BS_Quest,error=FALSE,warning=FALSE,results='asis'----------

mycoeffnames<-c("Period","SpiteSum",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)1",
                "SpiteSum:myBs(variable, knots = c(50), degree = 1)2",
                "Risk_overall",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)1",
                "Risk_overall:myBs(variable, knots = c(50), degree = 1)2",
                            "GeschlechtMale","rivalry","SVO","I(IA + IAS)",
                            "(Intercept)","Observations","Log Likelihood","Akaike Inf. Crit",
                            "Bayesian Inf. Crit.")
ModelNames<-c("$C'_1$","$C'_2$","$C'_3$","$C'_4$","$C'_5$")
newnames=c( "Period","Spite","\\mbs{Spite}{\\vLow}","\\mbs{Spite}{\\vHigh}",
            "Risk","\\mbs{Risk}{\\vLow}","\\mbs{Risk}{\\vHigh}","Male", "Rivalry","SVO","IA","Constant")


PrintstargazerGam(list(Model1=Model_Over1_Gam_BS_Quest,Model2=Model_Over2_Gam_BS_Quest,Model3=Model_Over3_Gam_BS_Quest,
                       Model4=Model_Over4_Gam_BS_Quest,Model5=Model_Over5_Gam_BS_Quest),
             multipleObjects=TRUE,
             mycoeffnames=mycoeffnames,
             ModelNames=ModelNames,newnames=newnames
             )




## ----FSPcompetingSpiteMeasures------------------------------------------------
dataCompeteFPA<-within(data_for_testFSP,{Kim<-scale(Spite_Kim); SVO<-scale(Spite_SVO); Quest<-scale(Spite_Quest)})
dataCompeteSPA<-within(data_for_test   ,{Kim<-scale(Spite_Kim); SVO<-scale(Spite_SVO); Quest<-scale(Spite_Quest)})
Model2FSPCompete<-lmer(overbid ~ (Kim+SVO+Quest)*variable+(1|sid) +Period+(1|gid),data=dataCompeteFPA)
coeffsCompFPA<-fixef(Model2FSPCompete)
coeffsCompFPA<-data.frame(beta=coeffsCompFPA[grep(":variable",names(coeffsCompFPA))])
coeffsCompFPA[["type"]]<-sub(":variable","",rownames(coeffsCompFPA))
coeffsCompFPA[["vType"]]<-0

## ----competeSPA---------------------------------------------------------------
estCompeteSPA<-gamm(overbid ~ Quest + SVO + Kim +s(variable,bs="tp") #default
                       +(Quest + SVO + Kim):myBs(variable,knots=c(50),degree=1)
                       +Period,random=list(sid=~1,gid=~1),
                       data=dataCompeteSPA)
feCompeteSPA<-fixef(estCompeteSPA[["lme"]])
coeffsCompSPA<-data.frame(beta=feCompeteSPA[grep("myBs\\(variable",names(feCompeteSPA))])
coeffsCompSPA[["type"]]<-gsub("^X|:.*","",rownames(coeffsCompSPA))
coeffsCompSPA[["vType"]]<-as.numeric(gsub(".*\\)","",rownames(coeffsCompSPA)))


## ----competeFiga,external=FALSE,include=TRUE----------------------------------
spiteTran<-read.csv(text="type,typex
Kim,\\begin{tabular}{c}Kimbrough\\\\[-.5ex]Reiss\\end{tabular}
SVO,\\begin{tabular}{c}Own\\\\[-.5ex]measure\\end{tabular}
Quest,Marcus et al.")
vTran<-read.csv(sep=";",text="vType;vLab
0;Spite$\\times v$
1;Spite$\\times v[0,50]$
2;Spite$\\times v[50,100]$")
coeffsComp<-within(join(join(rbind.fill(coeffsCompSPA),spiteTran),vTran),
    aType<-ifelse(vType==0,"\\Fpa{}","\\Iipap"))
dotplot(beta ~ reorder(typex,beta)| aType,group=vLab,ylab="$\\beta$",data=coeffsComp,scales=list(y="free"),#layout=c(2,1),
        par.strip.text=list(cex=.8),
        auto.key=list(cex=.7,corner=c(1,0),border=1,background="#BBBBBB")) + layer(panel.refline(h=0))


## ----save.image,cache=FALSE---------------------------------------------------
##save.image("all.Rdata")
save(file="reviewer.Rdata",optAR)

